/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.client.sse;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.sse.client.SseListener;

import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;

public class SseStreamConsumer implements Runnable {

  private static final Logger LOGGER = getLogger(SseStreamConsumer.class);

  private final HttpEntity entity;
  private final ServerSentEventDecoder decoder;

  public SseStreamConsumer(HttpEntity entity, SseListener listener, String clientName) {
    this.entity = entity;
    this.decoder = new ServerSentEventDecoder(listener);
    LOGGER.trace("Creating SSE stream consumer with decoder id '{}' associated to client named '{}'", decoder.getId(),
                 clientName);
  }

  @Override
  public void run() {
    InputStream content = entity.getContent();

    try (decoder) {
      if (entity.getBytesLength().isPresent()) {
        int length = (int) entity.getBytesLength().getAsLong();
        byte[] buffer = new byte[length];
        int actualRead = content.read(buffer);
        decoder.decode(buffer, actualRead);
      } else {
        byte[] buffer = new byte[8096];
        boolean eosReached = false;
        while (!eosReached) {
          int actualRead = content.read(buffer);
          if (actualRead == -1) {
            eosReached = true;
          } else {
            decoder.decode(buffer, actualRead);
          }
        }
      }
    } catch (IOException e) {
      throw new RuntimeException("Failed while consuming SSE stream of decoder with id %d".formatted(decoder.getId()), e);
    }
  }
}
