/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalToCompressingWhiteSpace;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;

import java.io.IOException;

import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Issue("W-17413024")
class ServerReadTimeoutTestCase extends AbstractHttpServerTestCase {

  private static final long SMALL_READ_TIMEOUT_MILLIS = 100L;

  public ServerReadTimeoutTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "test-server";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder.setReadTimeout(SMALL_READ_TIMEOUT_MILLIS);
  }

  @BeforeEach
  public void setup() throws Exception {
    setUpServer();

    server.addRequestHandler("/test", (requestContext, responseCallback) -> {
      String asString;
      try {
        asString = IOUtils.toString(requestContext.getRequest().getEntity().getContent(), UTF_8);
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
      responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(asString)),
                                     new NoOpResponseStatusCallback());
    });
  }

  @Test
  void sendPartialRequestShouldTimeout() throws IOException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", port)) {
      // Send only one chunk...
      tcpTextClient.sendString("""
          POST /test HTTP/1.1\r
          Host: localhost: %d\r
          Transfer-Encoding: chunked\r
          \r
          7\r
          Partial\r
          """.formatted(port));

      // The response should be received normally.
      String header = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(header, equalToCompressingWhiteSpace("""
          HTTP/1.1 408 Request Timeout\r
          Connection: close\r
          Content-Length: 23"""));

      String content = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(content, is("Timeout reading request"));

      String nothing = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(nothing, is(""));
    }
  }
}
