/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.service.http.test.netty.utils.TestUtils.preservingHeaderCase;
import static org.mule.service.http.test.netty.utils.TestUtils.sendRawRequestToServer;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class ServerHeaderCaseSensitivityTestCase extends AbstractHttpServerTestCase {

  private static final String A_RAW_REQUEST_WITH_HEADERS = """
      GET /saveRequest HTTP/1.1
      Host: localhost
      User-Agent: ObviouslyMe
      HeaderWithCase: HeaderValue

      """;

  public ServerHeaderCaseSensitivityTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "test-server";
  }

  private final BlockingQueue<HttpRequest> receivedRequests = new LinkedBlockingQueue<>();

  @BeforeEach
  void setup() throws Exception {
    setUpServer();
    server.addRequestHandler("/saveRequest", (requestContext, responseCallback) -> {
      receivedRequests.add(requestContext.getRequest());
      responseCallback.responseReady(HttpResponse.builder().addHeader("HeaderWithCase", "HeaderValue").build(),
                                     new NoOpResponseStatusCallback());
    }).start();
  }

  @Test
  void receivedRequestLosesHeaderCaseWhenNoPropertySet() throws IOException, InterruptedException {
    String rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", port);

    HttpRequest requestSeenByServer = receivedRequests.take();
    assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("host", "user-agent", "headerwithcase"));

    assertThat(rawResponse, containsString("headerwithcase"));
  }

  @Test
  void receivedRequestPreserveHeaderCaseWhenPropertySet() {
    preservingHeaderCase(() -> {
      String rawResponse;
      try {
        rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", port);
        HttpRequest requestSeenByServer = receivedRequests.take();
        assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("Host", "User-Agent", "HeaderWithCase"));
      } catch (IOException | InterruptedException e) {
        throw new RuntimeException(e);
      }

      assertThat(rawResponse, containsString("HeaderWithCase"));
    });
  }
}
