/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static io.netty.handler.codec.http.HttpMethod.GET;
import static io.netty.handler.codec.http.HttpResponseStatus.BAD_REQUEST;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static io.netty.util.CharsetUtil.UTF_8;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.service.http.netty.impl.server.RejectFailedUpgradeHandler;
import org.mule.service.http.netty.impl.server.RejectFailedUpgradeHandler.FailedHttpUpgrade;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpRequest;
import org.junit.jupiter.api.Test;

/**
 * Unit tests for {@link RejectFailedUpgradeHandler}.
 */
class RejectFailedUpgradeHandlerTestCase extends AbstractHttpTestCase {

  /** It verifies that normal HTTP requests pass through the handler unchanged. */
  @Test
  void testNormalRequestPassesThrough() {
    EmbeddedChannel channel = new EmbeddedChannel(new RejectFailedUpgradeHandler());

    HttpRequest request = new DefaultHttpRequest(HTTP_1_1, GET, "/test");

    channel.writeInbound(request);
    HttpRequest inboundRequest = channel.readInbound();

    assertThat(inboundRequest, is(request));
    assertThat(channel.isOpen(), is(true));
  }

  /**
   * It verifies that failed upgrade requests trigger a 400 BAD_REQUEST response and close the connection.
   */
  @Test
  void testFailedUpgradeSendsBadRequestAndClosesConnection() {
    EmbeddedChannel channel = new EmbeddedChannel(new RejectFailedUpgradeHandler());

    DefaultFullHttpRequest upgradeRequest = new DefaultFullHttpRequest(HTTP_1_1, GET, "/upgrade");
    String errorMessage = "Upgrade failed";

    FailedHttpUpgrade failedUpgrade = new FailedHttpUpgrade(upgradeRequest, errorMessage);
    channel.pipeline().fireUserEventTriggered(failedUpgrade);

    FullHttpResponse response = channel.readOutbound();

    assertThat(response.status(), is(BAD_REQUEST));
    assertThat(response.content().toString(UTF_8), is(errorMessage));
    assertThat(channel.isOpen(), is(false));
  }
}

