/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;

import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;

import java.io.IOException;

import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Issue("W-17408113")
public class HalfTcpShutdownTestCase extends AbstractHttpServerTestCase {

  public HalfTcpShutdownTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setup() throws Exception {
    setUpServer();
    server.addRequestHandler("/test", (requestContext, responseCallback) -> {
      try {
        var asString = IOUtils.toString(requestContext.getRequest().getEntity().getContent(), UTF_8);
        responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(asString)),
                                       new NoOpResponseStatusCallback());
      } catch (IOException e) {
        responseCallback.responseReady(new ResponseWithoutHeaders(INTERNAL_SERVER_ERROR, new StringHttpEntity(e.toString())),
                                       new NoOpResponseStatusCallback());
      }
    });
  }

  @Override
  protected String getServerName() {
    return "test-server";
  }

  @Test
  void halfShutdownClient() throws IOException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", port)) {
      // Send a chunked request...
      tcpTextClient.sendString("""
          POST /test HTTP/1.1\r
          Host: localhost: %d\r
          Transfer-Encoding: chunked\r
          \r
          2\r
          OK\r
          0\r
          \r
          """.formatted(port));

      // And shutdown output
      tcpTextClient.shutdownWrite();

      // The response should be received normally.
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), containsString("content-length: 2"));
      assertThat(tcpTextClient.receive(2), containsString("OK"));
    }
  }
}
