/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.classloader;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.tck.junit4.matcher.Eventually.eventually;
import static org.mule.tck.util.CollectableReference.collectedByGc;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.server.TestHttpServer;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.util.CollectableReference;

import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Issue;

@Ignore("W-17091411")
public class NettyClientResourceReleaserTestCase extends AbstractHttpTestCase {

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");
  @Rule
  public TestHttpServer testServer = new TestHttpServer("localhost", serverPort.getNumber(), false);
  private HttpClient client;


  @Before
  public void setUp() throws Exception {
    client = NettyHttpClient.builder().withUsingPersistentConnections(true).build();
    client.start();
    testServer.addRequestHandler("/hello", (request, responseSender) -> {
      HttpEntity helloContent = new StringHttpEntity("Hello from server!");
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();
  }


  @Test
  @Issue("W-15631454")
  public void applicationClassLoaderShouldBeCollectedByGcAfterClientIsClosed()
      throws IOException, InterruptedException, ExecutionException {
    HttpRequest httpRequest = HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("GET")
        .entity(new EmptyHttpEntity()).build();

    ClassLoader parentClassLoader = currentThread().getContextClassLoader();
    CollectableReference<ClassLoader> classLoaderRef =
        new CollectableReference<>(new URLClassLoader(new URL[0], parentClassLoader));

    final var tccl = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(classLoaderRef.get());
      Future<HttpResponse> response = client.sendAsync(httpRequest);
      String responseContent = IOUtils.toString(response.get().getEntity().getContent(), UTF_8);
      assertThat(responseContent, is("Hello from server!"));
      client.stop();
    } finally {
      currentThread().setContextClassLoader(tccl);
    }
    assertThat(classLoaderRef, is(eventually(collectedByGc())));
  }
}
