/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.MULTIPART;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;
import org.mule.runtime.http.api.domain.entity.multipart.MultipartHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import org.apache.commons.io.IOUtils;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import io.qameta.allure.Story;

@Story(MULTIPART)
public class HttpServerMixedPartTestCase extends AbstractHttpServerTestCase {

  private static final String MIXED_CONTENT =
      "--the-boundary\r\n"
          + "Content-Type: text/plain\r\n"
          + "Custom: myHeader\r\n"
          + "Content-Disposition: attachment; name=\"field\"\r\n"
          + "\r\n"
          + "My data here\r\n"
          + "--the-boundary--\r\n";

  public HttpServerMixedPartTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  public void setUp() throws Exception {
    setUpServer();
    server.addRequestHandler("/", (requestContext, responseCallback) -> {
      String data = "My data here";
      HttpPart part = new HttpPart("field", data.getBytes(), TEXT.toRfcString(), data.length());
      part.addHeader("Custom", "myHeader");
      responseCallback.responseReady(HttpResponse.builder().entity(new MultipartHttpEntity(singletonList(part)))
          .addHeader(CONTENT_TYPE, MULTIPART_MIXED.toRfcString() + "; boundary=\"the-boundary\"")
          .build(), new IgnoreResponseStatusCallback());

    });
  }

  @Override
  protected String getServerName() {
    return "parts-test";
  }

  @Test
  void handlesMultipartMixed() throws Exception {
    try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
      HttpGet httpPost = new HttpGet("http://localhost:" + port);
      try (ClassicHttpResponse response = httpClient.execute(httpPost)) {
        assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8), is(equalTo(MIXED_CONTENT)));
      }
    }
  }


}
