/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.sse;

import static org.mule.runtime.http.api.sse.client.SseSourceConfig.fromResponse;
import static org.mule.runtime.http.api.sse.server.SseClientConfig.builderFrom;
import static org.mule.service.http.test.common.util.HttpEntityMatcher.hasContent;
import static org.mule.service.http.test.common.util.HttpMessageHeaderMatcher.hasHeader;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.STREAMABLE_HTTP;
import static org.mule.service.http.test.netty.utils.sse.ServerSentEventTypeSafeMatcher.aServerSentEvent;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.sse.SSEEventsAggregator;
import org.mule.tck.junit5.DynamicPort;

import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;


@Story(STREAMABLE_HTTP)
@Issue("W-18238331")
public class StreamableHttpTestCase extends AbstractHttpServiceTestCase {

  private static final String TEST_DATA_EVENT_1 = "{'method': 'blah'}";
  private static final String TEST_DATA_EVENT_2 = "{'method': 'bleh'}";
  private static final String TEST_JSON = "{'method': 'blih'}";
  public static final String DEFAULT_EVENT_TYPE = "message";

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private HttpServer httpServer;
  private HttpClient httpClient;

  public StreamableHttpTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() throws Exception {
    httpServer = service.getServerFactory().create(new HttpServerConfiguration.Builder()
        .setName("Streamable HTTP Server")
        .setHost("localhost")
        .setPort(serverPort)
        .build());
    httpServer.start();
    httpServer.addRequestHandler("/streamable-http", (ctx, callback) -> {
      try {
        if (shouldRespondWithSSE(ctx.getRequest())) {
          // If you want to start an SSE response, you can use startSseResponse that returns an SseClient:
          var clientConfig = builderFrom(ctx)
              .withClientId("my-client-id")
              .customizeResponse(customizer -> customizer.addResponseHeader("X-My-Custom-Header", "TheValue"))
              .build();
          try (var sseClient = callback.startSseResponse(clientConfig)) {
            sseClient.sendEvent(TEST_DATA_EVENT_1);
            sseClient.sendEvent(TEST_DATA_EVENT_2);
          }
        } else {
          // If you want to return a JSON, use the old known mechanism:
          HttpResponse response = HttpResponse.builder()
              .addHeader("Content-Type", "application/json")
              .entity(new StringHttpEntity(TEST_JSON))
              .build();
          callback.responseReady(response, new NoOpResponseStatusCallback());
        }
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    });

    httpClient = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("Streamable HTTP Client")
        .setStreaming(true)
        .build());
    httpClient.start();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
    httpServer.stop().dispose();
  }

  @Test
  void streamableHttpReturnsSSE() throws ExecutionException, InterruptedException {
    streamableHttpTest("SSE");
  }

  @Test
  void streamableHttpReturnsJson() throws ExecutionException, InterruptedException {
    streamableHttpTest("JSON");
  }

  private void streamableHttpTest(String testExpectedContentType) throws InterruptedException, ExecutionException {
    SSEEventsAggregator listener = new SSEEventsAggregator();

    String sseUri = "http://localhost:" + serverPort + "/streamable-http";
    HttpRequest request = HttpRequest.builder()
        .uri(sseUri)
        .addHeader("Accept", "application/json, text/event-stream")
        .addHeader("X-SubProtocol", testExpectedContentType)
        .build();

    var future = httpClient.sendAsync(request);

    // You can also use a whenComplete, but it's a test, so I want the assertions in the main thread.
    HttpResponse response = future.get();
    if (isSse(response)) {
      assertThat(response, hasHeader("X-My-Custom-Header", "TheValue"));
      try (var sseSource = httpClient.sseSource(fromResponse(response).build())) {
        sseSource.register(listener);
        sseSource.open();
        var eventsList = listener.getList();
        assertThat(eventsList, contains(aServerSentEvent(DEFAULT_EVENT_TYPE, TEST_DATA_EVENT_1),
                                        aServerSentEvent(DEFAULT_EVENT_TYPE, TEST_DATA_EVENT_2)));
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    } else {
      assertThat(response.getEntity(), hasContent(equalTo(TEST_JSON)));
    }
  }

  private static boolean isSse(HttpResponse response) {
    return response.getHeaderValue("Content-Type").equals("text/event-stream");
  }

  // This is the test logic to differentiate whether we want SSE or JSON.
  private boolean shouldRespondWithSSE(HttpRequest request) {
    String expectedSubprotocol = request.getHeaderValue("X-SubProtocol");
    return "SSE".equalsIgnoreCase(expectedSubprotocol);
  }
}

