/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.http.api.HttpHeaders.Values.CLOSE;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.hc.core5.http.HttpVersion.HTTP_1_0;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalToCompressingWhiteSpace;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;

import org.apache.hc.core5.http.HttpVersion;
import org.hamcrest.Matcher;
import org.junit.jupiter.api.Test;

public class HttpServerTransfer10TestCase extends HttpServerTransferTestCase {

  public static final String CRLF = "\r\n";

  public HttpServerTransfer10TestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  public HttpVersion getVersion() {
    return HTTP_1_0;
  }

  @Test
  void defaultsStreamingWhenEmpty() throws Exception {
    verify10Headers(EMPTY, CLOSE, is(nullValue()), EMPTY);
  }

  @Test
  void defaultsStreamingWhenBytes() throws Exception {
    verify10Headers(BYTES, CLOSE, is(nullValue()), DATA);
  }

  @Test
  void defaultsStreamingWhenMultipart() throws Exception {
    verify10Headers(MULTIPART, CLOSE, is(nullValue()), MULTIPART_DATA);
  }

  @Test
  void defaultsStreamingWhenStream() throws Exception {
    verify10Headers(STREAM, CLOSE, is(nullValue()), DATA);
  }

  private void verify10Headers(String path, String expectedConnection, Matcher<Object> contentLengthMatcher,
                               String expectedBody)
      throws IOException {
    // As apache client 5 doesn't work with HTTP/1.0, use raw TCP socket to send the HTTP/1.0 request...
    try (Socket socket = new Socket("localhost", port);
        PrintWriter out = new PrintWriter(socket.getOutputStream(), true);
        BufferedReader in = new BufferedReader(new InputStreamReader(socket.getInputStream(), UTF_8))) {

      // Send HTTP/1.0 request
      String requestPath = path.isEmpty() ? "/" : path;
      out.println("GET " + requestPath + " HTTP/1.0");
      out.println("Host: localhost:" + port);
      out.println("Connection: close");
      out.println(); // Empty line to end headers

      // Read response
      String statusLine = in.readLine();
      assertThat("Status line should indicate HTTP/1.0 and 200", statusLine.startsWith("HTTP/1.0 200"), is(true));

      // Parse headers
      String contentLengthValue = null;
      String connectionValue = null;
      String headerLine;
      while ((headerLine = in.readLine()) != null && !headerLine.isEmpty()) {
        String[] parts = headerLine.split(":", 2);
        if (parts.length == 2) {
          String headerName = parts[0].trim().toLowerCase();
          if (headerName.equals("content-length")) {
            contentLengthValue = parts[1].trim();
          }
          if (headerName.equals("connection")) {
            connectionValue = parts[1].trim();
          }
        }
      }

      // Read body
      StringBuilder body = new StringBuilder();
      String line;
      while ((line = in.readLine()) != null) {
        if (body.length() != 0) {
          body.append(CRLF);
        }
        body.append(line);
      }

      assertThat(connectionValue, is(expectedConnection));
      assertThat(contentLengthValue, contentLengthMatcher);
      assertThat(body.toString(), equalToCompressingWhiteSpace(expectedBody));
    }
  }
}
