/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.client.sse;

import static java.lang.Thread.currentThread;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.sse.ServerSentEvent;
import org.mule.runtime.http.api.sse.client.SseListener;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import org.slf4j.Logger;

/**
 * Implementation of {@link ProgressiveBodyDataListener} that parses instances of {@link ServerSentEvent} and calls a
 * {@link SseListener} for each parsed instance.
 */
public class ServerSentEventDataListener implements ProgressiveBodyDataListener {

  private static final Logger LOGGER = getLogger(ServerSentEventDataListener.class);

  private final CompletableFuture<InputStream> inputStreamFuture = new CompletableFuture<>();
  private final ServerSentEventDecoder decoder;

  public ServerSentEventDataListener(SseListener eventListener, String clientName) {
    this.decoder = new ServerSentEventDecoder(eventListener);
    LOGGER.trace("Creating SSE Data listener with decoder id '{}' associated to client named '{}'", decoder.getId(), clientName);
  }

  @Override
  public void onStreamCreated(InputStream inputStream) {
    if (inputStreamFuture.isDone()) {
      throw new IllegalStateException("Another input stream was already added to the event decoder");
    }
    LOGGER.trace("SSE Listener ({}): Event stream created.", decoder.getId());
    inputStreamFuture.complete(inputStream);
  }

  /**
   * This method is called when a number of bytes is available in the stream. It must not be called concurrently.
   *
   * @param newDataLength the newly available number of bytes.
   */
  @Override
  public void onDataAvailable(int newDataLength) {
    if (newDataLength <= 0) {
      return;
    }

    InputStream inputStream = getInputStream();

    // TODO (W-18041071): Optimize this parsing to avoid data duplication.
    LOGGER.trace("SSE Listener ({}): {} bytes available in the stream. Reading...", decoder.getId(), newDataLength);
    byte[] data = new byte[newDataLength];
    try {
      int actualRead = inputStream.read(data);
      if (actualRead != newDataLength) {
        LOGGER.warn("SSE Listener ({}): Expected to read {} bytes, but actually read {}", decoder.getId(), newDataLength,
                    actualRead);
      }
      decoder.decode(data, actualRead);
    } catch (IOException e) {
      throw new IllegalStateException("SSE Listener (%d): Exception parsing events stream".formatted(decoder.getId()), e);
    }
  }

  private InputStream getInputStream() {
    try {
      return inputStreamFuture.get();
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new IllegalStateException("SSE Listener (%d): Interrupted while getting stream".formatted(decoder.getId()), e);
    } catch (ExecutionException e) {
      throw new IllegalStateException("SSE Listener (%d): Execution error while getting stream".formatted(decoder.getId()),
                                      e.getCause());
    }
  }

  @Override
  public void onEndOfStream() {
    LOGGER.trace("SSE Listener ({}): End of the stream. Closing...", decoder.getId());
    decoder.close();
  }
}
