/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;
import org.mule.service.http.netty.impl.server.util.HttpParser;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Collection;

import io.qameta.allure.Issue;
import jakarta.mail.BodyPart;
import jakarta.mail.MessagingException;
import org.junit.Test;

public class HttpParserTestCase extends AbstractHttpTestCase {

  @Test
  @Issue("W-17843326")
  public void testParseMultipartContent_validContent() throws IOException {
    String contentType = "multipart/mixed; boundary=boundary";
    String multipartContent =
        "--boundary\r\nContent-Disposition: form-data; name=\"part1\"\r\n\r\npart1 content\r\n--boundary\r\nContent-Disposition: form-data; name=\"part2\"\r\n\r\npart2 content\r\n--boundary--";

    ByteArrayInputStream inputStream = new ByteArrayInputStream(multipartContent.getBytes());

    Collection<HttpPart> parts = HttpParser.parseMultipartContent(inputStream, contentType);

    assertThat(parts.size(), is(2));
    HttpPart part1 = parts.stream().filter(part -> part.getName().equals("part1")).findFirst().orElse(null);
    assertThat(part1, is(notNullValue()));
    assertThat(part1.getName(), is("part1"));
    HttpPart part2 = parts.stream().filter(part -> part.getName().equals("part2")).findFirst().orElse(null);
    assertThat(part2, is(notNullValue()));
    assertThat(part2.getName(), is("part2"));

  }

  @Test
  @Issue("W-17843326")
  public void testParseMultipartContent_invalidContentType() {
    String invalidContentType = "text/plain";
    String invalidMultipartContent = "Invalid content";

    ByteArrayInputStream inputStream = new ByteArrayInputStream(invalidMultipartContent.getBytes());
    assertThrows(IOException.class, () -> HttpParser.parseMultipartContent(inputStream, invalidContentType));
  }

  @Test
  @Issue("W-17843326")
  public void testParseMultipartContent_missingBoundary() {
    String contentType = "multipart/mixed";
    String missingBoundaryContent = "No boundary specified";

    ByteArrayInputStream inputStream = new ByteArrayInputStream(missingBoundaryContent.getBytes());

    assertThrows(IOException.class, () -> HttpParser.parseMultipartContent(inputStream, contentType));
  }

  @Test
  @Issue("W-17843326")
  public void testParseMultipartContent_withMultipartRelatedAndContentId() throws IOException, MessagingException {
    String contentType = "multipart/related; boundary=boundary";
    String multipartContent =
        "--boundary\r\nContent-Disposition: form-data; name=\"part1\"\r\n\r\npart1 content\r\n--boundary\r\nContent-Disposition: form-data; name=\"part2\"\r\n\r\npart2 content\r\n--boundary--";
    BodyPart mockPart = mock(BodyPart.class);
    when(mockPart.getFileName()).thenReturn(null);
    when(mockPart.getContentType()).thenReturn("multipart/related; boundary=boundary");
    when(mockPart.getHeader("Content-ID")).thenReturn(new String[] {"part123"});
    ByteArrayInputStream inputStream = new ByteArrayInputStream(multipartContent.getBytes());

    Collection<HttpPart> parts = HttpParser.parseMultipartContent(inputStream, contentType);

    assertThat(parts.size(), is(2));
    HttpPart part = parts.stream().filter(p -> p.getName().equals("part1")).findFirst().orElse(null);
    assertThat(part, is(notNullValue()));
    assertThat(part.getName(), is("part1"));
  }
}
