/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.util.Preconditions.checkArgument;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.service.http.netty.impl.streaming.ResponseStatusCallbackAdapter;
import org.mule.service.http.netty.impl.streaming.StreamingEntitySender;

import java.io.IOException;
import java.util.concurrent.Executor;

import io.netty.channel.ChannelHandlerContext;

/**
 * Implementation of {@link ResponseSender} that sends the response entity in a streamed fashion. It reads a chunk of the entity's
 * content input stream and sends it, and defers a promise to repeat the same process. Once there is no more data in the input
 * stream, it sends an empty last content and closes the input stream.
 */
public class StreamingResponseSender extends BaseResponseSender {

  private final StreamingEntitySender entitySender;

  public StreamingResponseSender(HttpRequest request, ChannelHandlerContext ctx, HttpResponse response,
                                 ResponseStatusCallback statusCallback, Executor ioExecutor, HttpWriter writer) {
    super(request, ctx, response, new ResponseStatusCallbackAdapter(statusCallback), writer);
    checkArgument(response.getEntity().isStreaming(), "Response entity must be streaming to use a StreamingResponseSender");
    requireNonNull(response.getEntity().getContent(), "Response content cannot be null");
    entitySender = new StreamingEntitySender(response.getEntity(), ctx, this::sendHeaderIfNeeded,
                                             new ResponseStatusCallbackAdapter(statusCallback),
                                             ioExecutor, writer);
  }

  @Override
  protected void sendContent() throws IOException {
    entitySender.sendNextChunk();
  }
}
