/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.transport;

import static java.lang.Boolean.getBoolean;

import java.util.concurrent.Executor;

import io.netty.channel.EventLoopGroup;
import io.netty.channel.ServerChannel;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;

/**
 * Utility class that centralizes the places where a resource is created for EPoll or NIO (and eventually for KQueue) depending on
 * its availability.
 * <p>
 * It only tries to return that native implementation if the system property {@code mule.http.service.tryUseNative} is set to
 * {@code true}.
 */
public final class NativeChannelTransportSpecifics {

  private static final String TRY_USING_NATIVE_PROP = "mule.http.service.tryUseNative";
  private static final boolean TRY_USING_NATIVE = getBoolean(TRY_USING_NATIVE_PROP);

  private NativeChannelTransportSpecifics() {
    // private empty to avoid wrong instantiations.
  }

  /**
   * Creates a {@link EpollEventLoopGroup} or a {@link NioEventLoopGroup} depending on the epoll availability and the values of
   * the system property {@link #TRY_USING_NATIVE_PROP}.
   *
   * @param threadsCount number of threads of the resulting {@link EventLoopGroup}.
   * @param executor     where the event loops have to be executed.
   * @return the corresponding {@link EventLoopGroup}.
   */
  public static EventLoopGroup createLoopGroup(int threadsCount, Executor executor) {
    if (TRY_USING_NATIVE && Epoll.isAvailable()) {
      return new EpollEventLoopGroup(threadsCount, executor);
    }
    return new NioEventLoopGroup(threadsCount, executor);
  }

  /**
   * @return the class that should be used for datagram channels, depending on the epoll availability and the values of the system
   *         property {@link #TRY_USING_NATIVE_PROP}.
   */
  public static Class<? extends DatagramChannel> getDatagramChannelType() {
    if (TRY_USING_NATIVE && Epoll.isAvailable()) {
      return EpollDatagramChannel.class;
    }
    return NioDatagramChannel.class;
  }

  /**
   * @return the class that should be used for server channels, depending on the epoll availability and the values of the system
   *         property {@link #TRY_USING_NATIVE_PROP}.
   */
  public static Class<? extends ServerChannel> getServerChannelType() {
    if (TRY_USING_NATIVE && Epoll.isAvailable()) {
      return EpollServerSocketChannel.class;
    }
    return NioServerSocketChannel.class;
  }

  /**
   * @return whether this class will try to use native implementations.
   */
  public static boolean triesUsingNative() {
    return TRY_USING_NATIVE;
  }
}
