/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static org.mule.runtime.api.scheduler.SchedulerConfig.config;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.service.http.netty.impl.util.SslContextHelper.sslContextForClient;

import static java.lang.Integer.getInteger;
import static java.lang.Integer.max;
import static java.lang.Runtime.getRuntime;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

public class HttpClientConnectionManager implements ContextHttpClientConnectionFactory, Disposable, Initialisable {

  private static final int CLIENT_SELECTOR_THREAD_COUNT =
      getInteger("mule.http.client.selectors.count", max(getRuntime().availableProcessors(), 2));

  private final SchedulerService schedulerService;
  private final Map<String, HttpClient> clients = new ConcurrentHashMap<>();
  private Scheduler ioTasksScheduler;

  public HttpClientConnectionManager(SchedulerService schedulerService) {
    this.schedulerService = schedulerService;
  }

  public HttpClient getOrCreateClient(String name, Supplier<? extends HttpClientConfiguration> configuration) {
    checkArgument(name != null, "Client name can't be null");
    return clients.computeIfAbsent(name, serverName -> create(configuration.get()));
  }

  @Override
  public HttpClient create(HttpClientConfiguration configuration) {
    try {
      checkArgument(configuration != null, "Client configuration can't be null");
      // TODO: Handle the other parameters:
      // configuration.getResponseBufferSize()
      NettyHttpClient.Builder builder = NettyHttpClient.builder()
          .withName(configuration.getName())
          .withProxyConfig(configuration.getProxyConfig())
          .withSslContext(sslContextForClient(
                                              configuration.getTlsContextFactory(),
                                              configuration.getHttp1ProtocolConfig().isEnabled(),
                                              configuration.getHttp2ProtocolConfig().isEnabled()))
          .withConnectionIdleTimeout(configuration.getConnectionIdleTimeout())
          .withUsingPersistentConnections(configuration.isUsePersistentConnections())
          .withSelectorsScheduler(this::createSelectorsScheduler)
          .withSelectorsCount(CLIENT_SELECTOR_THREAD_COUNT)
          .withMaxConnections(configuration.getMaxConnections())
          .withIOTasksScheduler(ioTasksScheduler)
          .withResponseStreamingEnabled(configuration.isStreaming())
          .withTcpProperties(getTcpProperties(configuration))
          .withHttp1Config(configuration.getHttp1ProtocolConfig())
          .withHttp2Config(configuration.getHttp2ProtocolConfig())
          .withDecompressionEnabled(configuration.isDecompress());
      return enrichClientBuilder(builder, configuration).build();
    } catch (IllegalArgumentException e) {
      throw new MuleRuntimeException(e);
    }
  }

  private TcpClientSocketProperties getTcpProperties(HttpClientConfiguration configuration) {
    var configProperties = configuration.getClientSocketProperties();
    if (configProperties != null) {
      return configProperties;
    } else {
      return TcpClientSocketProperties.builder().build();
    }
  }

  private Scheduler createSelectorsScheduler() {
    if (schedulerService == null) {
      return null;
    }
    return schedulerService
        .customScheduler(config()
            .withDirectRunCpuLightWhenTargetBusy(true)
            .withMaxConcurrentTasks(CLIENT_SELECTOR_THREAD_COUNT)
            .withName("http.requester"), CLIENT_SELECTOR_THREAD_COUNT);
  }

  private Scheduler getIOSchedulerIfPossible() {
    if (schedulerService == null) {
      return null;
    }
    return schedulerService.ioScheduler();
  }

  protected NettyHttpClient.Builder enrichClientBuilder(NettyHttpClient.Builder builder,
                                                        HttpClientConfiguration configuration) {
    return builder;
  }

  @Override
  public void dispose() {
    if (ioTasksScheduler != null) {
      ioTasksScheduler.shutdown();
      ioTasksScheduler = null;
    }
  }

  @Override
  public void initialise() throws InitialisationException {
    ioTasksScheduler = getIOSchedulerIfPossible();
  }
}
