/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.runtime.api.util.DataUnit.KB;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.STREAMING;
import static org.mule.tck.probe.PollingProber.probe;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.slf4j.LoggerFactory.getLogger;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;

import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;

import io.qameta.allure.Description;
import io.qameta.allure.Story;

@Story(STREAMING)
@DisplayName("Validates cases in streaming where POST bodies are consumed more than once")
public abstract class HttpClientPostStreamingTestCase extends AbstractHttpClientTestCase {

  private static final Logger LOGGER = getLogger(HttpClientPostStreamingTestCase.class);

  public static final int RESPONSE_TIMEOUT = 3000;

  private String payloadAfterDancing;

  private HttpClientConfiguration.Builder clientBuilder = new HttpClientConfiguration.Builder().setName("streaming-test");

  public HttpClientPostStreamingTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Test
  @Description("Verifies that in streaming the redirection preserves the post body and the request stream is not consumed on redirect without reset")
  void redirectionPreservesPostBody() throws Exception {
    HttpClient client =
        service.getClientFactory().create(clientBuilder.setResponseBufferSize(KB.toBytes(10)).setStreaming(true).build());
    client.start();
    try {
      HttpRequestOptions options = getOptions();
      HttpRequest request = getRequest();

      Latch responseReceivedLatch = new Latch();
      client.sendAsync(request, options).whenComplete((response, exception) -> responseReceivedLatch.release());
      responseReceivedLatch.await(RESPONSE_TIMEOUT, MILLISECONDS);

      probe(() -> {
        assertThat(payloadAfterDancing, is(expectedPayload()));
        return true;
      });
    } finally {
      client.stop();
    }
  }

  protected String expectedPayload() {
    return TEST_PAYLOAD;
  }


  public abstract HttpRequest getRequest();

  public abstract HttpRequestOptions getOptions();

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    return doSetUpHttpResponse(request);
  }

  public abstract HttpResponse doSetUpHttpResponse(HttpRequest request);


  protected void extractPayload(HttpRequest request) {
    try {
      payloadAfterDancing = new String(request.getEntity().getBytes());
    } catch (IOException e) {
      LOGGER.debug("Could not extract payload.");
    }
  }

}
