/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static java.util.Collections.singleton;

import static jakarta.servlet.http.HttpServletResponse.SC_OK;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.DummyRequestHandler;
import org.mule.service.http.test.netty.utils.client.TestSSLNettyClientWithBouncyCastle;
import org.mule.service.http.test.netty.utils.server.TestHttpServer;
import org.mule.tck.junit4.rule.DynamicPort;

import java.security.Security;

import io.qameta.allure.Issue;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

public class NettyHttpServerWithBouncyCastleTestCase extends AbstractHttpTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public TestHttpServer testServer = new TestHttpServer("localhost", serverPort.getNumber(), true);

  @Rule
  public TestSSLNettyClientWithBouncyCastle testClient =
      new TestSSLNettyClientWithBouncyCastle("localhost", serverPort.getNumber());

  @Before
  public void setup() {
    Security.addProvider(new BouncyCastleProvider());

    testServer.addRequestHandler("/path", new DummyRequestHandler());
    testServer.addRequestHandler(singleton("GET"), "/only-get", new DummyRequestHandler());
  }

  @After
  public void tearDown() {
    Security.removeProvider("BC");
  }

  @Test
  @Issue("W-15631497")
  public void testSSLConnection() throws Exception {
    String sslEndpoint = "/path";
    HttpResponse response = testClient.sendGet(sslEndpoint);
    assertThat("Expected response status code to be 200", response.getStatusCode(), is(SC_OK));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLRehandshake() throws Exception {
    String sslEndpoint = "/path";
    HttpResponse initialResponse = testClient.sendGet(sslEndpoint);
    assertThat(initialResponse.getStatusCode(), is(SC_OK));
    testClient.reHandshake();
    HttpResponse rehandshakeResponse = testClient.sendGet(sslEndpoint);
    assertThat(rehandshakeResponse.getStatusCode(), is(SC_OK));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLHandshakeFailureWhenKeystorePasswordIsIncorrect() {
    try {
      testClient.sendGet("/path");
    } catch (Exception e) {
      assertThat("Unable to initialise TLS configuration", is(e.getMessage()));
    }
  }

  @Test
  @Issue("W-15631497")
  public void testSSLHandshakeFailureWhenTrustStorePasswordIsIncorrect() {
    try {
      testClient.sendGet("/path");
    } catch (Exception e) {
      assertThat("Unable to initialise TLS configuration", is(e.getMessage()));
    }
  }
}
