/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static io.netty.buffer.ByteBufUtil.getBytes;
import static io.netty.buffer.Unpooled.wrappedBuffer;
import static io.netty.handler.codec.http.HttpMethod.GET;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static io.netty.handler.codec.http.LastHttpContent.EMPTY_LAST_CONTENT;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;

import org.mule.service.http.netty.impl.server.ConditionalRequestAggregator;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.handler.codec.DecoderResult;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.DefaultLastHttpContent;
import io.netty.handler.codec.http.FullHttpRequest;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.Issue;

@Issue("W-19285006")
class ConditionalRequestAggregatorTestCase extends AbstractHttpTestCase {

  private final EmbeddedChannel channelWithAggregator = new EmbeddedChannel(new ConditionalRequestAggregator());

  @Test
  void aggregateWithEmptyContent() {
    var headerPart = new DefaultHttpRequest(HTTP_1_1, GET, "https://localhost:80/");

    channelWithAggregator.writeInbound(headerPart, EMPTY_LAST_CONTENT);
    channelWithAggregator.flush();

    FullHttpRequest fullRequest = channelWithAggregator.readInbound();
    assertThat(fullRequest.method(), is(GET));
    assertThat(fullRequest.protocolVersion(), is(HTTP_1_1));
    assertThat(fullRequest.uri(), is("https://localhost:80/"));

    assertThat(getBytes(fullRequest.content()), is(new byte[0]));
  }

  @Test
  void aggregateWithNonEmptyContent() {
    var headerPart = new DefaultHttpRequest(HTTP_1_1, GET, "https://localhost:80/");
    var contentBytes = "Hello, World!".getBytes();
    var lastContent = new DefaultLastHttpContent(wrappedBuffer(contentBytes));
    channelWithAggregator.writeInbound(headerPart, lastContent);
    channelWithAggregator.flush();

    FullHttpRequest fullRequest = channelWithAggregator.readInbound();

    assertThat(fullRequest.method(), is(GET));
    assertThat(fullRequest.protocolVersion(), is(HTTP_1_1));
    assertThat(fullRequest.uri(), is("https://localhost:80/"));
    assertThat(getBytes(fullRequest.content()), is(contentBytes));
  }

  @Test
  void inboundFullRequestRemainsUntouched() {
    var original = new DefaultFullHttpRequest(HTTP_1_1, GET, "https://localhost:80/");
    channelWithAggregator.writeInbound(original);
    channelWithAggregator.flush();

    FullHttpRequest fullRequest = channelWithAggregator.readInbound();
    assertThat(fullRequest, sameInstance(original));
  }

  @Test
  void failureRequestRemainsUntouched() {
    var original = new DefaultHttpRequest(HTTP_1_1, GET, "https://localhost:80/");
    original.setDecoderResult(DecoderResult.failure(new RuntimeException("Expected!")));

    channelWithAggregator.writeInbound(original);
    channelWithAggregator.flush();

    var afterTheAggregator = channelWithAggregator.readInbound();
    assertThat(afterTheAggregator, sameInstance(original));
  }

  @Test
  void threePartsRequestIsNotAConcernOfTheAggregator() {
    var headerPart = new DefaultHttpRequest(HTTP_1_1, GET, "https://localhost:80/");
    var actualContent = new DefaultHttpContent(wrappedBuffer("Hello, World!".getBytes()));
    channelWithAggregator.writeInbound(headerPart, actualContent, EMPTY_LAST_CONTENT);
    channelWithAggregator.flush();

    var first = channelWithAggregator.readInbound();
    var second = channelWithAggregator.readInbound();
    var third = channelWithAggregator.readInbound();

    assertThat(first, sameInstance(headerPart));
    assertThat(second, sameInstance(actualContent));
    assertThat(third, sameInstance(EMPTY_LAST_CONTENT));
  }
}
