/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static java.util.Arrays.asList;
import static java.util.List.of;

import static io.netty.handler.ssl.ApplicationProtocolNames.HTTP_1_1;
import static io.netty.handler.ssl.ApplicationProtocolNames.HTTP_2;
import static io.netty.handler.ssl.IdentityCipherSuiteFilter.INSTANCE_DEFAULTING_TO_SUPPORTED_CIPHERS;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.tls.TlsContextFactory;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.List;

import io.netty.handler.ssl.ApplicationProtocolConfig;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.JdkSslContext;
import io.netty.handler.ssl.SslContext;

public final class SslContextHelper {

  private SslContextHelper() {
    // empty to avoid unintentional instantiations
  }

  public static SslContext sslContextForServer(TlsContextFactory tlsContextFactory,
                                               boolean allowHttp1,
                                               boolean allowHttp2)
      throws NoSuchAlgorithmException, KeyManagementException {
    if (tlsContextFactory == null) {
      return null;
    }

    ClientAuth clientAuth = tlsContextFactory.isTrustStoreConfigured() ? ClientAuth.REQUIRE : ClientAuth.OPTIONAL;
    return new JdkSslContext(tlsContextFactory.createSslContext(),
                             false,
                             getCiphers(tlsContextFactory),
                             INSTANCE_DEFAULTING_TO_SUPPORTED_CIPHERS,
                             new ApplicationProtocolConfig(
                                                           ApplicationProtocolConfig.Protocol.ALPN,
                                                           // NO_ADVERTISE is currently the only mode supported by both
                                                           // OpenSsl and JDK providers.
                                                           ApplicationProtocolConfig.SelectorFailureBehavior.NO_ADVERTISE,
                                                           // ACCEPT is currently the only mode supported by both OpenSsl
                                                           // and JDK providers.
                                                           ApplicationProtocolConfig.SelectedListenerFailureBehavior.FATAL_ALERT,
                                                           getProtocols(allowHttp1, allowHttp2)),
                             clientAuth, tlsContextFactory.getEnabledProtocols(), false);
  }

  private static List<String> getProtocols(boolean allowHttp1, boolean allowHttp2) {
    if (allowHttp1 && allowHttp2) {
      return of(HTTP_2, HTTP_1_1);
    }

    if (allowHttp1) {
      return of(HTTP_1_1);
    }

    if (allowHttp2) {
      return of(HTTP_2);
    }

    return of();
  }

  public static SslContext sslContextForClient(TlsContextFactory tlsContextFactory, boolean allowHttp1, boolean allowHttp2) {
    if (tlsContextFactory == null) {
      return null;
    }
    try {
      return new JdkSslContext(tlsContextFactory.createSslContext(),
                               true,
                               getCiphers(tlsContextFactory),
                               INSTANCE_DEFAULTING_TO_SUPPORTED_CIPHERS,
                               new ApplicationProtocolConfig(
                                                             ApplicationProtocolConfig.Protocol.ALPN,
                                                             // NO_ADVERTISE is currently the only mode supported by both
                                                             // OpenSsl and JDK providers.
                                                             ApplicationProtocolConfig.SelectorFailureBehavior.NO_ADVERTISE,
                                                             // ACCEPT is currently the only mode supported by both OpenSsl
                                                             // and JDK providers.
                                                             ApplicationProtocolConfig.SelectedListenerFailureBehavior.FATAL_ALERT,
                                                             getProtocols(allowHttp1, allowHttp2)),
                               ClientAuth.NONE, tlsContextFactory.getEnabledProtocols(), false);
    } catch (KeyManagementException | NoSuchAlgorithmException e) {
      throw new MuleRuntimeException(e);
    }
  }

  private static Iterable<String> getCiphers(TlsContextFactory tlsContextFactory) {
    if (tlsContextFactory == null) {
      return null;
    }
    String[] enabledCipherSuites = tlsContextFactory.getEnabledCipherSuites();
    if (enabledCipherSuites == null) {
      return null;
    }
    return asList(enabledCipherSuites);
  }
}
