/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.service.http.netty.impl.server.ChannelPromises.promiseToCallback;

import static io.netty.buffer.Unpooled.EMPTY_BUFFER;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.runtime.http.api.sse.server.SseClient;
import org.mule.runtime.http.api.sse.server.SseClientConfig;
import org.mule.service.http.netty.impl.streaming.ResponseStatusCallbackAdapter;

import java.io.IOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.concurrent.Executor;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.Http2FrameStream;

public class NettyHttp2RequestReadyCallback implements HttpResponseReadyCallback {

  private final ChannelHandlerContext ctx;
  private final Http2FrameStream frameStream;
  private final HttpRequest request;
  private final Executor ioExecutor;
  private final Http2Writer frameWriter;

  public NettyHttp2RequestReadyCallback(ChannelHandlerContext ctx, Http2FrameStream stream, HttpRequest request,
                                        Executor ioExecutor) {
    this.ctx = ctx;
    this.frameStream = stream;
    this.request = request;
    this.ioExecutor = ioExecutor;
    this.frameWriter = new Http2Writer(ctx, frameStream);
  }

  @Override
  public void responseReady(HttpResponse response, ResponseStatusCallback responseStatusCallback) {
    try {
      sendResponse(ctx, response, responseStatusCallback);
    } catch (Exception e) {
      responseStatusCallback.onErrorSendingResponse(e);
    }
  }

  @Override
  public Writer startResponse(HttpResponse response, ResponseStatusCallback responseStatusCallback, Charset encoding) {
    var statusCallback = new ResponseStatusCallbackAdapter(responseStatusCallback);
    frameWriter.writeResponseHeader(request, response, promiseToCallback(ctx, statusCallback, false));
    return new Writer() {

      @Override
      public void write(char[] cbuf, int off, int len) {
        ByteBuf content = ctx.alloc().buffer();
        content.writeCharSequence(new String(cbuf, off, len), encoding);
        frameWriter.writeContent(content, false, promiseToCallback(ctx, statusCallback, false));
      }

      @Override
      public void flush() {
        ctx.flush();
      }

      @Override
      public void close() {
        frameWriter.writeContent(EMPTY_BUFFER, true, promiseToCallback(ctx, statusCallback, true));
      }
    };
  }

  @Override
  public SseClient startSseResponse(SseClientConfig config) {
    // TODO: If SSE over HTTP/2 is needed, check if this works:
    // return new SseResponseStarter().startResponse(config, this);
    throw new UnsupportedOperationException("SSE is not supported over HTTP/2");
  }

  private void sendResponse(ChannelHandlerContext ctx, HttpResponse response, ResponseStatusCallback callback)
      throws IOException {
    if (response.getEntity().isStreaming()) {
      new StreamingResponseSender(request, ctx, response, callback, ioExecutor, frameWriter).send();
    } else {
      new DirectResponseSender(request, ctx, response, callback, frameWriter).send();
    }
  }
}
