/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server.util;

import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpConstants.Method.POST;
import static org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistry.HTTP_SERVICE_ENCODED_SLASH_ENABLED_PROPERTY;
import static org.mule.tck.MuleTestUtils.testWithSystemProperty;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;

import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.server.MethodRequestMatcher;
import org.mule.runtime.http.api.server.PathAndMethodRequestMatcher;
import org.mule.runtime.http.api.utils.MatcherCollisionException;
import org.mule.runtime.http.api.utils.RequestMatcherRegistry;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistry;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistry.Path;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistry.RequestHandlerMatcherPair;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistryBuilder;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import org.hamcrest.Matcher;
import org.junit.Test;

public class RequestMatcherRegistryTestCase extends AbstractHttpTestCase {

  private static final Object METHOD_MISMATCH = new Object();
  private static final Object NOT_FOUND = new Object();
  private static final Object DISABLED = new Object();
  private static final Object SECOND_LEVEL_CATCH_ALL = new Object();
  private static final Object SECOND_LEVEL_SPECIFIC = new Object();
  private static final Object FIRST_LEVEL_SPECIFIC = new Object();
  private DefaultRequestMatcherRegistry<Object> registry;
  private Object handler;

  // Tests for DefaultRequestMatcherRegistry.add()
  @Test
  public void testAddPath() {
    // root path
    resetRegistryAndHandler();
    PathAndMethodRequestMatcher rootMatcher = PathAndMethodRequestMatcher.builder().path("/").build();
    registry.add(rootMatcher, handler);
    assertThat(registry.find(HttpRequest.builder().uri("http://localhost:8081/").method(GET).build()), is(handler));

    // wildcard path
    resetRegistryAndHandler();
    PathAndMethodRequestMatcher wildcardMatcher = PathAndMethodRequestMatcher.builder().path("/*").build();
    registry.add(wildcardMatcher, handler);
    assertThat(registry.find(HttpRequest.builder().uri("http://localhost:8081/any/path").method(GET).build()), is(handler));

    // specific path
    resetRegistryAndHandler();
    PathAndMethodRequestMatcher specificMatcher = PathAndMethodRequestMatcher.builder().path("/foo/bar").build();
    registry.add(specificMatcher, handler);
    assertThat(registry.find(HttpRequest.builder().uri("http://localhost:8081/foo/bar").method(GET).build()), is(handler));
  }

  @Test
  public void testAddMultipleMethods() {
    resetRegistryAndHandler();
    PathAndMethodRequestMatcher matcher = PathAndMethodRequestMatcher.builder()
        .path("/foo")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).add(POST).build())
        .build();
    registry.add(matcher, handler);

    HttpRequest getRequest = HttpRequest.builder().uri("http://localhost:8081/foo").method(GET).build();
    Object getResult = registry.find(getRequest);

    HttpRequest postRequest = HttpRequest.builder().uri("http://localhost:8081/foo").method(POST).build();
    Object postResult = registry.find(postRequest);

    assertThat(getResult, is(handler));
    assertThat(postResult, is(handler));
  }

  // Tests for DefaultRequestMatcherRegistry.find()
  @Test
  public void findByRequest() {
    registry = buildRegistry(getFullBuilder());
    validateRequestMatch(registry, "/path/somewhere", SECOND_LEVEL_CATCH_ALL);
    validateRequestMatch(registry, "/path/here", SECOND_LEVEL_SPECIFIC);
    validateRequestMatch(registry, "/here", DISABLED);
    validateRequestMatch(registry, "/nope", NOT_FOUND);
    validateRequestMatch(registry, "/path/somewhere", sameInstance(METHOD_MISMATCH), POST);
  }

  @Test
  public void allNullIfDefault() {
    registry = buildRegistry(new DefaultRequestMatcherRegistryBuilder<>());
    validateMethodAndPathMatch(registry, "/path/somewhere", SECOND_LEVEL_CATCH_ALL);
    validateMethodAndPathMatch(registry, "/path/here", SECOND_LEVEL_SPECIFIC);
    validateMethodAndPathMatch(registry, "/here", nullValue(), GET);
    validateMethodAndPathMatch(registry, "/nope", nullValue(), GET);
    validateMethodAndPathMatch(registry, "/path/somewhere", nullValue(), POST);
  }

  @Test
  public void testFindCatchAllWithMethod() {
    registry = new DefaultRequestMatcherRegistry<>();
    Object catchAllGetHandler = new Object();
    registry.add(PathAndMethodRequestMatcher.builder()
        .path("/*")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .build(), catchAllGetHandler);

    // matching method (GET)
    HttpRequest request1 = HttpRequest.builder().uri("http://localhost:8081/some/path").method(GET).build();
    Object result1 = registry.find(request1);
    assertThat(result1, is(catchAllGetHandler));

    // non-matching method (POST)
    HttpRequest request2 = HttpRequest.builder().uri("http://localhost:8081/some/path").method(POST).build();
    Object result2 = registry.find(request2);
    assertThat(result2, is(nullValue()));
  }

  @Test
  public void testFindRootPathWithTrailingSlash() {
    registry = buildRegistry(getFullBuilder());

    Object rootHandler = new Object();
    registry.add(PathAndMethodRequestMatcher.builder().path("/").build(), rootHandler);

    HttpRequest request = HttpRequest.builder().uri("http://localhost:8081/").method(GET).build();
    Object result = registry.find(request);

    assertThat(result, is(rootHandler));
  }

  // Tests for DefaultRequestMatcherRegistry.pathDecodedWithEncodedSlashes()
  @Test
  public void testPathDecodedWithEncodedSlashes() throws Exception {
    testWithSystemProperty(HTTP_SERVICE_ENCODED_SLASH_ENABLED_PROPERTY, "true", () -> {
      resetRegistryAndHandler();

      String encodedPath = "/foo%2Fbar%2Fbaz";
      String decodedPath = "/foo/bar/baz";

      registry.add(PathAndMethodRequestMatcher.builder().path(decodedPath).build(), handler);

      HttpRequest request = HttpRequest.builder().uri("http://localhost:8081" + encodedPath).method(GET).build();
      Object result = registry.find(request);
      assertThat(result, is(handler));
    });
  }

  @Test
  public void testPathDecodedWithEncodedSlashesWithNoEncodedSlashes() {
    resetRegistryAndHandler();
    String path = "/foo/bar/baz";
    registry.add(PathAndMethodRequestMatcher.builder().path(path).build(), handler);

    HttpRequest request = HttpRequest.builder().uri("http://localhost:8081" + path).method(GET).build();
    Object result = registry.find(request);
    assertThat(result, is(handler));
  }

  @Test
  public void testPathDecodedWithEncodedSlashesWithSingleEncodedSlash() throws Exception {
    testWithSystemProperty(HTTP_SERVICE_ENCODED_SLASH_ENABLED_PROPERTY, "true", () -> {
      resetRegistryAndHandler();

      String decodedPath = "/foo/bar";
      String encodedPath = "/foo%2Fbar";

      registry.add(PathAndMethodRequestMatcher.builder().path(decodedPath).build(), handler);

      HttpRequest request = HttpRequest.builder().uri("http://localhost:8081" + encodedPath).method(GET).build();
      Object result = registry.find(request);
      assertThat(result, is(handler));
    });
  }

  @Test
  public void testPathDecodedWithEncodedSlashesWithEncodedSlashes() throws Exception {
    testWithSystemProperty(HTTP_SERVICE_ENCODED_SLASH_ENABLED_PROPERTY, "true", () -> {
      registry = new DefaultRequestMatcherRegistry<>();

      String encodedPath = "/foo%2Fbar%2Fbaz";
      String decodedPath = registry.pathDecodedWithEncodedSlashes(encodedPath);

      assertThat(decodedPath, is("/foo%2Fbar%2Fbaz"));
    });
  }

  // Tests for DefaultRequestMatcherRegistry.remove()
  @Test
  public void testRemoveCatchAllPath() {
    registry = buildRegistry(getFullBuilder());

    RequestMatcherRegistry.RequestMatcherRegistryEntry entry =
        registry.add(PathAndMethodRequestMatcher.builder().path("/foo/*").build(), new Object());
    entry.remove();

    validateRequestMatch(registry, "/foo/bar", NOT_FOUND);
  }

  @Test
  public void testRemoveUriParameterPath() {
    registry = buildRegistry(getFullBuilder());

    RequestMatcherRegistry.RequestMatcherRegistryEntry entry =
        registry.add(PathAndMethodRequestMatcher.builder().path("/foo/{param}").build(), new Object());
    entry.remove();

    validateRequestMatch(registry, "/foo/bar", NOT_FOUND);
  }

  @Test
  public void testRemoveFromCatchAllUriParam() throws Exception {
    Path<Object> path = new Path<>("test", null);
    PathAndMethodRequestMatcher matcher = PathAndMethodRequestMatcher.builder().path("/{param}").build();
    var pair = new RequestHandlerMatcherPair<>(matcher, new Object());

    path.addChildPath("{param}", new Path<>("{param}", path));
    path.getCatchAllUriParam().addRequestHandlerMatcherPair(pair);

    boolean removed = path.removeRequestHandlerMatcherPair(pair);

    assertThat(removed, is(true));
    assertThat(path.getCatchAllUriParam(), is(nullValue()));
  }

  // Tests for DefaultRequestMatcherRegistry.Path
  @Test
  public void testRemoveChildPath() {
    Path<Object> root = new Path<>("root", null);
    Path<Object> child = new Path<>("child", root);

    root.addChildPath("child", child);
    root.removeChildPath("child");

    assertThat(root.getSubPaths().containsKey("child"), is(false));
  }

  @Test
  public void testMatchesNextSubPaths() {
    Path<Object> path = new Path<>("root", null);

    // add a URI parameter path with a child
    path.addChildPath("{param}", new Path<>("{param}", path));
    path.getCatchAllUriParam().addChildPath("other",
                                            new Path<>("other", path.getCatchAllUriParam()));

    // add a sub path that doesn't have the "other" child
    path.addChildPath("segment1", new Path<>("segment1", path));

    // call getChildPath with "segment1" (which has no "other" child) and "other"
    Path result = path.getChildPath("segment1", "other");

    // assert that the result is the catchAllUriParam (as matchesNextSubPaths will return false)
    assertThat(result, is(path.getCatchAllUriParam()));
  }

  // Other tests
  @Test
  public void testDefaultConstructor() {
    registry = new DefaultRequestMatcherRegistry<>();
    assertThat(registry, notNullValue());
  }

  @Test(expected = MatcherCollisionException.class)
  public void testValidateCollision() {
    registry = new DefaultRequestMatcherRegistry<>();
    PathAndMethodRequestMatcher matcher1 = PathAndMethodRequestMatcher.builder().path("/test").build();
    registry.add(matcher1, new Object());
    PathAndMethodRequestMatcher matcher2 = PathAndMethodRequestMatcher.builder().path("/test").build();
    registry.add(matcher2, new Object()); // throws MatcherCollisionException
  }

  @Test(expected = MatcherCollisionException.class)
  public void testCatchAllVsCatchAllSameMethod() {
    registry = new DefaultRequestMatcherRegistry<>();

    // add a catch-all matcher with GET
    registry.add(PathAndMethodRequestMatcher.builder()
        .path("/*")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .build(), new Object());

    // add another catch-all matcher with GET (should collide)
    registry.add(PathAndMethodRequestMatcher.builder()
        .path("/*")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .build(), new Object());
  }

  @Test(expected = MatcherCollisionException.class)
  public void testUriParameterVsUriParameter() {
    registry = new DefaultRequestMatcherRegistry<>();

    // add a URI parameter matcher with GET
    registry.add(PathAndMethodRequestMatcher.builder()
        .path("/{param1}")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .build(), new Object());

    // add another URI parameter matcher with GET (should collide)
    registry.add(PathAndMethodRequestMatcher.builder()
        .path("/{param2}")
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .build(), new Object());
  }

  private DefaultRequestMatcherRegistry<Object> buildRegistry(DefaultRequestMatcherRegistryBuilder<Object> builder) {
    registry = (DefaultRequestMatcherRegistry<Object>) builder.build();
    registry.add(PathAndMethodRequestMatcher.builder()
        .methodRequestMatcher(MethodRequestMatcher.builder().add(GET).build())
        .path("/path/*")
        .build(),
                 SECOND_LEVEL_CATCH_ALL);
    registry.add(PathAndMethodRequestMatcher.builder().path("/path/here").build(), SECOND_LEVEL_SPECIFIC);
    registry.add(PathAndMethodRequestMatcher.builder().path("/here").build(), FIRST_LEVEL_SPECIFIC).disable();
    return registry;
  }

  private DefaultRequestMatcherRegistryBuilder<Object> getFullBuilder() {
    return (DefaultRequestMatcherRegistryBuilder<Object>) new DefaultRequestMatcherRegistryBuilder<>()
        .onMethodMismatch(() -> METHOD_MISMATCH)
        .onNotFound(() -> NOT_FOUND)
        .onDisabled(() -> DISABLED);
  }

  private void validateRequestMatch(DefaultRequestMatcherRegistry<Object> registry, String path, Object expectedItem) {
    validateRequestMatch(registry, path, sameInstance(expectedItem), GET);
  }

  private void validateRequestMatch(DefaultRequestMatcherRegistry<Object> registry, String path, Matcher matcher,
                                    HttpConstants.Method method) {
    assertThat(registry.find(HttpRequest.builder().uri(format("http://localhost:8081%s", path)).method(method).build()),
               is(matcher));
  }

  private void validateMethodAndPathMatch(DefaultRequestMatcherRegistry<Object> registry, String path, Object expectedItem) {
    validateMethodAndPathMatch(registry, path, sameInstance(expectedItem), GET);
  }

  private void validateMethodAndPathMatch(DefaultRequestMatcherRegistry<Object> registry, String path, Matcher matcher,
                                          HttpConstants.Method method) {
    assertThat(registry.find(method.name(), path), is(matcher));
  }

  private void resetRegistryAndHandler() {
    registry = new DefaultRequestMatcherRegistry<>();
    handler = new Object();
  }
}
