/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;
import static org.mule.service.http.test.netty.utils.TestUtils.preservingHeaderCase;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.tck.junit4.rule.DynamicPort;

import java.util.List;
import java.util.concurrent.ExecutionException;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class ClientHeaderCaseSensitivityTestCase extends AbstractHttpTestCase {

  private HttpClient client;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public HardcodedResponseTcpServer server = new HardcodedResponseTcpServer(serverPort.getNumber());

  @Before
  public void setUp() {
    client = NettyHttpClient.builder().build();
    client.start();

    server.setResponse("HTTP/1.1 200 OK\n" +
        "Content-Length: 0\n" +
        "HeaderWithCase: HeaderValue\n" +
        "\n");
  }

  @Test
  public void responseHeaderDoesntLoseCaseWhenPropertyIsSet() {
    preservingHeaderCase(() -> {
      HttpRequest httpRequest = HttpRequest.builder(PRESERVE_HEADER_CASE)
          .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
          .method("GET")
          .entity(new EmptyHttpEntity())
          .build();

      HttpResponse response;
      try {
        response = client.sendAsync(httpRequest).get();
      } catch (InterruptedException | ExecutionException e) {
        throw new RuntimeException(e);
      }
      assertThat(response.getStatusCode(), is(OK.getStatusCode()));
      assertThat(response.getHeaders().keySet(), containsInAnyOrder("Content-Length", "HeaderWithCase"));
    });
  }

  @Test
  public void responseHeaderLoseCaseWhenPropertyIsNotSet() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder(PRESERVE_HEADER_CASE)
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(response.getHeaders().keySet(), containsInAnyOrder("content-length", "headerwithcase"));
  }

  @Test
  public void requestHeaderDoesNotLoseCaseWhenPropertyIsSet() {
    preservingHeaderCase(() -> {
      HttpRequest httpRequest = HttpRequest.builder(PRESERVE_HEADER_CASE)
          .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
          .method("GET")
          .addHeader("HeaderWithCase", "HeaderValue")
          .entity(new EmptyHttpEntity())
          .build();

      try {
        HttpResponse ignored = client.sendAsync(httpRequest).get();
      } catch (InterruptedException | ExecutionException e) {
        throw new RuntimeException(e);
      }
      List<String> rawRequests = server.getReceivedRawRequests();
      assertThat(rawRequests, hasSize(1));
      String rawRequest = rawRequests.get(0);
      assertThat(rawRequest, containsString("HeaderWithCase"));
    });
  }

  @Test
  public void requestHeaderLoseCaseWhenPropertyIsNotSet() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder(PRESERVE_HEADER_CASE)
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("GET")
        .addHeader("HeaderWithCase", "HeaderValue")
        .entity(new EmptyHttpEntity())
        .build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    assertThat(rawRequest, containsString("headerwithcase"));
  }
}
