/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.REQUEST_TOO_LONG;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.service.http.netty.impl.server.NettyHttp1ResponseReadyCallback.refreshMaxServerResponseHeaders;
import static org.mule.service.http.netty.impl.server.util.HttpListenerRegistry.refreshMaxServerRequestHeaders;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.HEADERS;
import static org.mule.tck.junit4.rule.SystemProperty.callWithProperty;

import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;

import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.tck.junit5.DynamicPort;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.apache.hc.client5.http.fluent.Request;
import org.apache.hc.core5.http.NoHttpResponseException;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;

@Story(HEADERS)
public class HttpServiceMaxHeadersTestCase extends AbstractHttpServiceTestCase {

  @DynamicPort(systemProperty = "port")
  Integer port;

  private static final String SIMPLE_ENDPOINT = "/test";
  private static final String PAYLOAD1 = "p1";

  public HttpServiceMaxHeadersTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  protected String getServerName() {
    return "max-headers-test";
  }

  private void registerHandler(HttpConstants.Method httpMethod, String endpoint, String payload, HttpServer httpServer) {
    httpServer.addRequestHandler(singletonList(httpMethod.name()), ensureSlash(endpoint), (requestContext, responseCallback) -> {
      responseCallback.responseReady(HttpResponse.builder().entity(new ByteArrayHttpEntity(payload.getBytes()))
          .addHeader(CONTENT_TYPE, TEXT.toRfcString())
          .build(), new IgnoreResponseStatusCallback());
    });
  }

  @AfterEach
  public void tearDown() {
    refreshMaxServerRequestHeaders();
    refreshMaxServerResponseHeaders();
  }

  @Issue("MULE-19837")
  @Description("When the max number of request headers are set by System Properties, they should be " +
      "assigned correctly. If this max number is exceeded, a 413 status code should be returned.")
  @Test
  void whenRequestHasMoreHeadersThanMaxNumberThen413ShouldBeReturned() throws Throwable {
    HttpServer httpServer = callWithProperty(SYSTEM_PROPERTY_PREFIX + "http.MAX_SERVER_REQUEST_HEADERS", "3",
                                             this::refreshSystemPropertiesAndCreateServer);
    registerHandler(GET, SIMPLE_ENDPOINT, PAYLOAD1, httpServer);

    Request request = Request.get(urlForPath(httpServer, SIMPLE_ENDPOINT));
    request.addHeader("header1", "someValue");
    request.addHeader("header2", "someValue");
    request.addHeader("header3", "someValue");
    request.addHeader("header4", "someValue");

    org.apache.hc.core5.http.HttpResponse response = request.execute().returnResponse();

    assertThat(response.getCode(), is(REQUEST_TOO_LONG.getStatusCode()));

    httpServer.stop();
    httpServer.dispose();
  }

  @Issue("MULE-19837")
  @Description("When the max number of response headers are set by System Properties, they should be " +
      "assigned correctly. If this max number is exceeded, a NoHttpResponseException should be thrown")
  @Test
  void whenResponseHasMoreHeadersThanMaxNumberThenExceptionShouldBeThrown() throws Throwable {
    HttpServer httpServer = callWithProperty(SYSTEM_PROPERTY_PREFIX + "http.MAX_SERVER_RESPONSE_HEADERS", "0",
                                             this::refreshSystemPropertiesAndCreateServer);
    registerHandler(GET, SIMPLE_ENDPOINT, PAYLOAD1, httpServer);
    Request request = Request.get(urlForPath(httpServer, SIMPLE_ENDPOINT));
    try {
      assertThrows(NoHttpResponseException.class, request::execute);
    } finally {
      httpServer.stop();
      httpServer.dispose();
    }
  }

  protected HttpServer refreshSystemPropertiesAndCreateServer() throws Exception {
    refreshMaxServerRequestHeaders();
    refreshMaxServerResponseHeaders();
    HttpServer httpServer = service.getServerFactory().create(new HttpServerConfiguration.Builder()
        .setHost("localhost")
        .setPort(port)
        .setName(getServerName())
        .build());
    httpServer.start();
    return httpServer;
  }
}
