/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.test.netty.AllureConstants.HTTP_2;
import static org.mule.service.http.test.netty.utils.TestUtils.createSslContext;

import static java.lang.Thread.sleep;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static reactor.netty.http.HttpProtocol.H2;
import static reactor.netty.http.HttpProtocol.HTTP11;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.tck.junit5.DynamicPort;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;

import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.ChannelOutboundHandlerAdapter;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.ssl.SslCloseCompletionEvent;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslHandler;
import io.qameta.allure.Feature;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import reactor.core.publisher.Mono;
import reactor.netty.Connection;
import reactor.netty.DisposableServer;
import reactor.netty.NettyPipeline;
import reactor.netty.http.server.HttpServer;

@Feature(HTTP_2)
class Http2ClientServerCloseNotifyTestCase extends AbstractHttpServiceTestCase {

  private static final int CLOSE_NOTIFY_TIMEOUT_MS = 10000;

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private DisposableServer httpServer;
  private HttpClient httpClient;

  private final CountDownLatch closeNotifySent = new CountDownLatch(1);

  public Http2ClientServerCloseNotifyTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() throws Exception {
    httpServer = createServer();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
    httpServer.disposeNow();
  }

  @Test
  void http1Only_whenCloseNotifyIsReceivedThenConnectionIsClosed()
      throws ExecutionException, InterruptedException, CreateException {
    httpClient = createClient(true, false);
    whenCloseNotifyIsReceivedThenConnectionIsClosed();
  }

  @Test
  void http2Only_whenCloseNotifyIsReceivedThenConnectionIsClosed()
      throws ExecutionException, InterruptedException, CreateException {
    httpClient = createClient(false, true);
    whenCloseNotifyIsReceivedThenConnectionIsClosed();
  }

  @Test
  void bothProtocols_whenCloseNotifyIsReceivedThenConnectionIsClosed()
      throws ExecutionException, InterruptedException, CreateException {
    httpClient = createClient(true, true);
    whenCloseNotifyIsReceivedThenConnectionIsClosed();
  }

  // we can't make this a parameterized test case because it already extends from a parameterized test case
  private void whenCloseNotifyIsReceivedThenConnectionIsClosed() throws ExecutionException, InterruptedException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    httpClient.sendAsync(request).get();
    assertThat("close notify was never sent by the server", closeNotifySent.await(CLOSE_NOTIFY_TIMEOUT_MS, MILLISECONDS),
               is(true));

    // Not ideal but it is the only way we have to give it some time to actually receive and process the event through the
    // pipeline
    sleep(500);

    // Without the fix this one fails with java.io.IOException: Remotely closed
    // With the fix it should succeed by sending the request over a new connection
    httpClient.sendAsync(request).get();
  }

  private HttpClient createClient(boolean http1Enabled, boolean http2Enabled) throws CreateException {
    var client = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("HTTP/2 Client")
        .setHttp1Config(new Http1ProtocolConfig(http1Enabled))
        .setHttp2Config(new Http2ProtocolConfig(http2Enabled))
        .setTlsContextFactory(TlsContextFactory.builder().trustStorePath("trustStore")
            .trustStorePassword("mulepassword").insecureTrustStore(true).build())
        .build());
    client.start();
    return client;
  }

  private DisposableServer createServer()
      throws NoSuchAlgorithmException, KeyManagementException, CreateException {
    TlsContextFactory tlsContextFactory = TlsContextFactory.builder()
        .keyStorePath("serverKeystore")
        .keyStorePassword("mulepassword").keyAlias("muleserver").keyPassword("mulepassword").keyStoreAlgorithm("PKIX")
        .build();
    SslContext serverSslContext = createSslContext(tlsContextFactory);

    AtomicReference<SslHandler> sslHandler = new AtomicReference<>();
    return HttpServer.create()
        .port(serverPort)
        .protocol(HTTP11, H2)
        .secure(spec -> spec.sslContext(serverSslContext))
        .doOnChannelInit((observer, channel, remoteAddress) -> {
          sslHandler.set(channel.pipeline().get(SslHandler.class));
        })
        .doOnConnection(conn -> {
          // will send a close_notify after the last response is sent, but won't close the connection
          SendCloseNotifyAfterLastResponseHandler.register(conn.channel(), sslHandler.get(), closeNotifySent);
          IgnoreCloseNotifyHandler.register(conn);
        })
        .handle((req, res) -> res.sendString(Mono.just("test")))
        .bindNow();
  }

  /**
   * Server Handler used to send a TLS close_notify after the server last response has been flushed. The close_notify is sent
   * without closing the connection.
   */
  private static final class SendCloseNotifyAfterLastResponseHandler extends ChannelOutboundHandlerAdapter {

    static final String NAME = "handler.send_close_notify_after_response";
    private final CountDownLatch latch;
    private final SslHandler sslHandler;

    SendCloseNotifyAfterLastResponseHandler(CountDownLatch latch, SslHandler sslHandler) {
      this.latch = latch;
      this.sslHandler = sslHandler;
    }

    static void register(Channel channel, SslHandler sslHandler, CountDownLatch latch) {
      SendCloseNotifyAfterLastResponseHandler handler = new SendCloseNotifyAfterLastResponseHandler(latch, sslHandler);
      channel.pipeline().addBefore(NettyPipeline.HttpTrafficHandler, NAME, handler);
    }

    @Override
    public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) {
      if (msg instanceof LastHttpContent || (msg instanceof Http2DataFrame && ((Http2DataFrame) msg).isEndStream())) {
        // closeOutbound sends a close_notify but don't close the connection.
        promise.addListener(future -> sslHandler.closeOutbound().addListener(f -> latch.countDown()));
      }
      ctx.write(msg, promise);
    }
  }

  /**
   * Handler used by secured servers which don't want to close client connection when receiving a client close_notify ack. The
   * handler is placed just before the ReactiveBridge (ChannelOperationsHandler), and will block any received
   * SslCloseCompletionEvent events. Hence, ChannelOperationsHandler won't get the close_notify ack, and won't close the channel.
   */
  static final class IgnoreCloseNotifyHandler extends ChannelInboundHandlerAdapter {

    static final String NAME = "handler.ignore_close_notify";

    static void register(Connection cnx) {
      cnx.channel().pipeline().addBefore(NettyPipeline.ReactiveBridge, NAME, new IgnoreCloseNotifyHandler());
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) {
      if (!(evt instanceof SslCloseCompletionEvent) || !((SslCloseCompletionEvent) evt).isSuccess()) {
        ctx.fireUserEventTriggered(evt);
      }
    }
  }
}
