/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.service.http.test.netty.AllureConstants.HTTP_SERVICE;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.LISTENERS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClients;
import org.junit.Before;
import org.junit.Test;

@Feature(HTTP_SERVICE)
@Story(LISTENERS)
@Issue("W-18538795")
public class RequestHandlerConsumingPayloadTestCase extends AbstractHttpServerTestCase {

  private final CompletableFuture<String> payloadReceivedByServer = new CompletableFuture<>();

  public RequestHandlerConsumingPayloadTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Before
  public void setUp() throws Exception {
    setUpServer();
    server.addRequestHandler("/", (reqCtx, callback) -> {
      try {
        // Read the full entity synchronously as part of the request handler
        var bytes = reqCtx.getRequest().getEntity().getBytes();
        payloadReceivedByServer.complete(new String(bytes));
        callback.responseReady(HttpResponse.builder().statusCode(200).build(), new IgnoreResponseStatusCallback());
      } catch (IOException e) {
        payloadReceivedByServer.completeExceptionally(e);
      }
    });
  }

  @Override
  protected String getServerName() {
    return "TestServer";
  }

  @Test
  public void consumePayloadSync() throws IOException, ExecutionException, InterruptedException {
    String payload = "Test Payload";

    try (var httpClient = HttpClients.createDefault()) {
      HttpPost httpPost = new HttpPost("http://localhost:" + port.getNumber());
      httpPost.setEntity(new StringEntity(payload));

      try (var response = httpClient.execute(httpPost)) {
        assertThat(response.getStatusLine().getStatusCode(), is(200));
        assertThat(payloadReceivedByServer.get(), is(equalTo(payload)));
      }
    }
  }
}
