/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.smb;

/**
 * Utility class for handling SMB (Server Message Block) protocol data.
 * <p>
 * This class provides methods to read and write integers in various byte sizes (2, 4, and 8 bytes) from and to byte arrays, which
 * is essential for processing SMB messages and NTLM authentication.
 * </p>
 * <p>
 * This implementation is based on the jcifs library, available at <a href="https://github.com/kevinboone/jcifs">jcifs GitHub
 * Repository</a>.
 * </p>
 */
public class SMBUtil {

  public static int readInt2(byte[] src, int srcIndex) {
    return (src[srcIndex] & 0xFF) + ((src[srcIndex + 1] & 0xFF) << 8);
  }

  public static void writeInt4(long val, byte[] dst, int dstIndex) {
    dst[dstIndex] = (byte) (val);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >> 8);
  }

  public static long readInt8(byte[] src, int srcIndex) {
    return (readInt4(src, srcIndex) & 0xFFFFFFFFL) + ((long) (readInt4(src, srcIndex + 4)) << 32);
  }

  public static int readInt4(byte[] src, int srcIndex) {
    return (src[srcIndex] & 0xFF) + ((src[srcIndex + 1] & 0xFF) << 8) + ((src[srcIndex + 2] & 0xFF) << 16)
        + ((src[srcIndex + 3] & 0xFF) << 24);
  }

  public static void writeInt2(long val, byte[] dst, int dstIndex) {
    dst[dstIndex] = (byte) (val);
    dst[++dstIndex] = (byte) (val >> 8);
  }

  public static void writeInt8(long val, byte[] dst, int dstIndex) {
    dst[dstIndex] = (byte) (val);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >>= 8);
    dst[++dstIndex] = (byte) (val >> 8);
  }
}
