/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.domain.HttpProtocol.HTTP_1_1;
import static org.mule.tck.junit4.matcher.Eventually.eventually;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.emptyString;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;

import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

@Issue("W-15867731")
public class ServerGracefulShutdownTestCase extends AbstractHttpTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  private HttpServer httpServer;
  private HttpListenerRegistry listenerRegistry;

  private NettyHttpClient httpClient;

  private final Latch requestHandlerStartedLatch = new Latch();
  private final Latch requestHandlerCanContinue = new Latch();

  // "Infinite" timeout, so that we have to close all the connections before the server stops. Otherwise, the test will hang.
  private long shutdownTimeoutMillis = 50000000000L;

  private Long getShutdownTimeout() {
    return shutdownTimeoutMillis;
  }

  @Before
  public void setup() throws Exception {
    listenerRegistry = new HttpListenerRegistry();

    httpServer = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withShutdownTimeout(this::getShutdownTimeout)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", true, 3000, 1000L,
                                                                           null, 300, executorRule.getExecutor()))
        .build();
    httpServer.start();
    httpServer.addRequestHandler("/with-latch", (requestContext, responseCallback) -> {
      requestHandlerStartedLatch.release();
      try {
        requestHandlerCanContinue.await();
      } catch (InterruptedException e) {
        fail("Latch await unexpectedly interrupted");
      }
      responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity("Test body")),
                                     new NoOpResponseStatusCallback());
    });

    httpServer.addRequestHandler("/without-latch", (requestContext, responseCallback) -> {
      responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity("Test body")),
                                     new NoOpResponseStatusCallback());
    });

    httpClient = NettyHttpClient.builder()
        .withUsingPersistentConnections(true)
        .build();
    httpClient.start();
  }

  @After
  public void tearDown() {
    httpClient.stop();
    if (!httpServer.isStopped()) {
      httpServer.stop();
    }
    httpServer.dispose();
  }

  @Test
  public void whenStoppingTheServerInTheMiddleOfARequestItEndsWithoutIssues() throws InterruptedException, ExecutionException {
    // Note: for this case, the HTTP Connector also adds a "Connection: close" header, but the HttpService doesn't.
    // That's why such assertion isn't here.
    HttpRequest request = HttpRequest.builder()
        .protocol(HTTP_1_1)
        .method("GET")
        .uri(format("http://localhost:%d/with-latch", serverPort.getNumber()))
        .build();
    CompletableFuture<HttpResponse> future = httpClient.sendAsync(request);
    requestHandlerStartedLatch.await();

    executorRule.getExecutor().submit(httpServer::stop);

    requestHandlerCanContinue.release();
    HttpResponse httpResponse = future.get();
    String bodyAsString = IOUtils.toString(httpResponse.getEntity().getContent());
    assertThat(bodyAsString, containsString("Test body"));
  }

  @Test
  public void canReuseAPersistentConnectionWhileStopping() throws IOException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", serverPort.getNumber())) {
      // Send a normal request
      tcpTextClient.sendString("""
          GET /without-latch HTTP/1.1
          Host: localhost:%d

          """.formatted(serverPort.getNumber()));
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), containsString("content-length: 9"));
      assertThat(tcpTextClient.receive(9), containsString("Test body"));

      executorRule.getExecutor().submit(httpServer::stop);
      assertThat(httpServer, is(eventually(stopping())));

      // Send the same request, but with the server in stopping state
      tcpTextClient.sendString("""
          GET /without-latch HTTP/1.1
          Host: localhost:%d

          """.formatted(serverPort.getNumber()));
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), containsString("content-length: 9"));
      assertThat(tcpTextClient.receive(9), containsString("Test body"));

      // The server is still stopping (not stopped)
      assertThat(httpServer, is(stopping()));
    }

    // And when we close the connection, the server finishes the stop and becomes stopped
    assertThat(httpServer, is(eventually(stopped())));
  }

  @Test
  public void stopOperationFinishesWithConnectionsInflightIfTheTimeoutIsElapsed() throws IOException {
    shutdownTimeoutMillis = 100l;
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", serverPort.getNumber())) {
      // Send a normal request
      tcpTextClient.sendString("""
          GET /without-latch HTTP/1.1
          Host: localhost:%d

          """.formatted(serverPort.getNumber()));
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), containsString("content-length: 9"));
      assertThat(tcpTextClient.receive(9), containsString("Test body"));

      // The socket is connected, but the stop finishes because the timeout will be elapsed
      httpServer.stop();
      assertThat(httpServer, is(eventually(stopped())));

      // As the socket is closed by the other end, trying to read something returns an end of stream (the implementation of this
      // test client returns an empty string in that case).
      assertThat(tcpTextClient.receive(1), is(emptyString()));
    }
  }

  private Matcher<HttpServer> stopped() {
    return new TypeSafeMatcher<>() {

      @Override
      public void describeTo(Description description) {
        description.appendText("Server is not stopped");
      }

      @Override
      public boolean matchesSafely(HttpServer httpServer) {
        return httpServer.isStopped();
      }
    };
  }

  private Matcher<HttpServer> stopping() {
    return new TypeSafeMatcher<>() {

      @Override
      public void describeTo(Description description) {
        description.appendText("Server is not stopping");
      }

      @Override
      public boolean matchesSafely(HttpServer httpServer) {
        return httpServer.isStopping();
      }
    };
  }
}
