/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsStringIgnoringCase;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.tck.junit4.rule.DynamicPort;

import java.util.List;
import java.util.concurrent.ExecutionException;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Issue;

public class ClientBodySemanticsTestCase extends AbstractHttpTestCase {

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public HardcodedResponseTcpServer server = new HardcodedResponseTcpServer(serverPort.getNumber());
  private HttpClient client;

  @Before
  public void setUp() {
    client = NettyHttpClient.builder().build();
    client.start();

    server.setResponse("""
        HTTP/1.1 200 OK
        Content-Length: 0

        """);
  }

  @After
  public void tearDown() {
    if (client != null) {
      client.stop();
    }
  }

  @Test
  @Issue("W-15631517")
  public void noContentLengthHeaderForGetWithoutBodySemantics() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("GET")
        .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is not present in the request
    assertThat(rawRequest, not(containsStringIgnoringCase("content-length: 0")));
  }

  @Test
  @Issue("W-15631517")
  public void noContentLengthHeaderForDeleteWithoutBodySemantics() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("DELETE")
            .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is not present in the request
    assertThat(rawRequest, containsStringIgnoringCase("content-length: 0"));
  }

  @Test
  @Issue("W-15631517")
  public void contentLengthHeaderForPostWithEmptyBody() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("POST")
            .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is present in the request for POST
    assertThat(rawRequest, containsStringIgnoringCase("content-length: 0"));
  }

  @Test
  @Issue("W-15631517")
  public void contentLengthHeaderForPutWithEmptyBody() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("PUT")
        .entity(new EmptyHttpEntity()) // Sending empty body
        .build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "Content-Length: 0" is present in the request for PUT, case-insensitively
    assertThat(rawRequest, containsStringIgnoringCase("content-length: 0"));
  }
}
