/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.runtime.http.api.HttpHeaders.Names.ACCEPT_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.GZIP;
import static org.mule.service.http.netty.impl.client.NettyHttpClient.refreshSystemProperties;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.CLIENT_BODY_DECOMPRESSION;
import static org.mule.tck.MuleTestUtils.testWithSystemProperty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.concurrent.TimeoutException;
import java.util.zip.GZIPOutputStream;

import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Test;

@Story(CLIENT_BODY_DECOMPRESSION)
public class HttpClientDecompressionTestCase extends AbstractHttpClientTestCase {

  private static final byte[] TEST_MESSAGE = "This is a regular message.".getBytes();
  private static final byte[] compressedData = compressWithGzip(TEST_MESSAGE);

  private HttpClient httpClient;

  public HttpClientDecompressionTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  protected void refreshPropertiesIfNeeded() {
    refreshSystemProperties();
  }

  @After
  public void stopClient() {
    if (httpClient != null) {
      httpClient.stop();
    }
    refreshPropertiesIfNeeded();
  }

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    HttpResponseBuilder builder = HttpResponse.builder().entity(new ByteArrayHttpEntity(compressedData));
    if (request.getHeaderValue(ACCEPT_ENCODING) != null) {
      builder.addHeader(CONTENT_ENCODING, GZIP);
    }
    return builder.build();
  }

  @Test
  public void decompressesDataWhenAcceptIsPresent_explicitConfig() throws IOException, TimeoutException {
    httpClient = createClient(true);
    var response = httpClient.send(createRequest(true));
    assertThat(response.getEntity().getBytes(), is(TEST_MESSAGE));
  }

  @Test
  public void doesNotDecompressDataWithoutAccept_explicitConfig() throws IOException, TimeoutException {
    httpClient = createClient(true);
    var response = httpClient.send(createRequest(false));
    assertThat(response.getEntity().getBytes(), is(compressedData));
  }

  @Test
  public void decompressesDataWhenAcceptIsPresent_withProperty() throws Exception {
    testWithSystemProperty("mule.http.client.decompress", "true", () -> {
      refreshPropertiesIfNeeded();
      httpClient = createClient(false);
      var response = httpClient.send(createRequest(true));
      assertThat(response.getEntity().getBytes(), is(TEST_MESSAGE));
    });
  }

  @Test
  public void doesNotDecompressDataWithoutAccept_withProperty() throws Exception {
    testWithSystemProperty("mule.http.client.decompress", "true", () -> {
      refreshPropertiesIfNeeded();
      httpClient = createClient(false);
      var response = httpClient.send(createRequest(false));
      assertThat(response.getEntity().getBytes(), is(compressedData));
    });
  }

  private HttpClient createClient(boolean isExplicitDecompress) {
    var name = isExplicitDecompress ? "explicit-decompress-client" : "decompress-with-property-client";
    var configBuilder = new HttpClientConfiguration.Builder().setName(name);
    if (isExplicitDecompress) {
      configBuilder = configBuilder.setDecompress(true);
    }
    var client = service.getClientFactory().create(configBuilder.build());
    client.start();
    return client;
  }

  private HttpRequest createRequest(boolean withAcceptEncodingGzip) {
    var builder = HttpRequest.builder().uri(getUri());
    if (withAcceptEncodingGzip) {
      builder.addHeader(ACCEPT_ENCODING, GZIP);
    }
    return builder.build();
  }

  private static byte[] compressWithGzip(byte[] uncompressedData) {
    try {
      ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
      GZIPOutputStream gzipOutputStream = new GZIPOutputStream(byteArrayOutputStream);
      gzipOutputStream.write(uncompressedData);
      gzipOutputStream.flush();
      gzipOutputStream.close();
      return byteArrayOutputStream.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
