/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.util;

import static org.mule.service.http.netty.impl.client.auth.ntlm.util.Crypto.getMD5;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * Implementation of the HMAC-T64 (Hash-based Message Authentication Code using the MD5 hashing algorithm). This class extends
 * {@link MessageDigest} and provides methods to compute HMAC-T64 using a specified secret key.
 *
 * <p>
 * This implementation is based on the jcifs library, available at:
 * <a href="https://github.com/codelibs/jcifs">https://github.com/codelibs/jcifs</a>
 * </p>
 *
 * @see <a href="https://github.com/codelibs/jcifs">jcifs on GitHub</a>
 */
class HMACT64 extends MessageDigest {

  /** Block length for HMAC. */
  private static final int BLOCK_LENGTH = 64;

  /** Inner padding byte. */
  private static final byte IPAD = (byte) 0x36;

  /** Outer padding byte. */
  private static final byte OPAD = (byte) 0x5c;

  /** Algorithm name for HMAC-T64. */
  public static final String HMACT64 = "HMACT64";

  /** MD5 MessageDigest instance used for hashing. */
  private final MessageDigest md5;

  /** Inner padding array. */
  private final byte[] ipad = new byte[BLOCK_LENGTH];

  /** Outer padding array. */
  private final byte[] opad = new byte[BLOCK_LENGTH];

  /**
   * Creates an HMACT64 instance using the provided secret key material.
   *
   * @param key The key material to use for hashing.
   * @throws NoSuchAlgorithmException if the MD5 algorithm is not available.
   */
  public HMACT64(byte[] key) throws NoSuchAlgorithmException {
    super(HMACT64);
    int length = Math.min(key.length, BLOCK_LENGTH);
    for (int i = 0; i < length; i++) {
      this.ipad[i] = (byte) (key[i] ^ IPAD);
      this.opad[i] = (byte) (key[i] ^ OPAD);
    }
    for (int i = length; i < BLOCK_LENGTH; i++) {
      this.ipad[i] = IPAD;
      this.opad[i] = OPAD;
    }

    this.md5 = getMD5();
    engineReset();
  }

  @Override
  protected byte[] engineDigest() {
    byte[] digest = this.md5.digest();
    this.md5.update(this.opad);
    return this.md5.digest(digest);
  }

  @Override
  protected int engineDigest(byte[] buf, int offset, int len) {
    byte[] digest = this.md5.digest();
    this.md5.update(this.opad);
    this.md5.update(digest);
    try {
      return this.md5.digest(buf, offset, len);
    } catch (Exception ex) {
      throw new IllegalStateException();
    }
  }

  @Override
  protected int engineGetDigestLength() {
    return this.md5.getDigestLength();
  }

  @Override
  protected void engineReset() {
    this.md5.reset();
    this.md5.update(this.ipad);
  }

  @Override
  protected void engineUpdate(byte b) {
    this.md5.update(b);
  }

  @Override
  protected void engineUpdate(byte[] input, int offset, int len) {
    this.md5.update(input, offset, len);
  }
}
