/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.av;

import static org.mule.service.http.netty.impl.client.auth.ntlm.smb.SMBUtil.readInt4;
import static org.mule.service.http.netty.impl.client.auth.ntlm.smb.SMBUtil.writeInt4;

/**
 * Represents the AV_PAIR structure for flags in NTLM authentication. This class extends {@code AvPair} to manage NTLM flags,
 * handling raw byte data and encoding/decoding flag values.
 *
 * <p>
 * This implementation is based on the jcifs library, available at:
 * <a href="https://github.com/codelibs/jcifs">https://github.com/codelibs/jcifs</a>
 * </p>
 *
 * @see <a href="https://github.com/codelibs/jcifs">jcifs on GitHub</a>
 */
public class AvFlags extends AvPair {

  /**
   * Constructs an instance of {@code AvFlags} with the specified raw byte data.
   *
   * @param raw the raw byte data representing the flags for NTLM authentication.
   */
  public AvFlags(byte[] raw) {
    super(MsvAvFlags, raw);
  }

  /**
   * Constructs an instance of {@code AvFlags} with the specified integer flag value.
   *
   * @param flags the integer value of the flags for NTLM authentication.
   */
  public AvFlags(int flags) {
    this(encode(flags));
  }

  /**
   * Retrieves the integer flag value from the raw byte data.
   *
   * @return the decoded integer value of the flags.
   */
  public int getFlags() {
    return readInt4(this.getRaw(), 0);
  }

  /**
   * Encodes the specified integer flags into a 4-byte array.
   *
   * @param flags the integer value of the flags to encode.
   * @return a byte array containing the encoded flags.
   */
  private static byte[] encode(int flags) {
    byte[] raw = new byte[4];
    writeInt4(flags, raw, 0);
    return raw;
  }
}
