/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;

import org.mule.service.http.netty.impl.client.RedirectMethodChangeHandler;

import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.util.AttributeKey;
import org.junit.Test;

/**
 * Unit tests for {@link RedirectMethodChangeHandler}.
 */
public class RedirectMethodChangeHandlerTestCase {

  // Attribute keys used by RedirectMethodChangeHandler
  private static final AttributeKey<Boolean> ALWAYS_SEND_BODY_KEY = AttributeKey.valueOf("ALWAYS_SEND_BODY");
  private static final AttributeKey<Boolean> REDIRECT_CHANGE_METHOD = AttributeKey.valueOf("REDIRECT_CHANGE_METHOD");

  @Test
  public void testNoRedirectMethodNotChanged() {
    EmbeddedChannel channel = new EmbeddedChannel(new RedirectMethodChangeHandler());
    channel.attr(REDIRECT_CHANGE_METHOD).set(Boolean.FALSE);

    HttpRequest request = new DefaultHttpRequest(HttpVersion.HTTP_1_1, HttpMethod.POST, "/test");
    request.headers().set("Content-Type", "application/json");
    request.headers().set("Content-Length", "100");

    channel.writeOutbound(request);
    HttpRequest outboundRequest = channel.readOutbound();
    assertThat(outboundRequest.method(), is(HttpMethod.POST));
    assertThat(outboundRequest.headers().contains("Content-Type"), is(true));
    assertThat(outboundRequest.headers().contains("Content-Length"), is(true));
  }

  @Test
  public void testRedirectMethodChangedToGet() {
    EmbeddedChannel channel = new EmbeddedChannel(new RedirectMethodChangeHandler());
    channel.attr(REDIRECT_CHANGE_METHOD).set(Boolean.TRUE);
    channel.attr(ALWAYS_SEND_BODY_KEY).set(Boolean.FALSE);

    HttpRequest request = new DefaultHttpRequest(HttpVersion.HTTP_1_1, HttpMethod.POST, "/test");
    request.headers().set("Content-Type", "application/json");
    request.headers().set("Content-Length", "100");

    channel.writeOutbound(request);
    HttpRequest outboundRequest = channel.readOutbound();
    assertThat(outboundRequest.method(), is(HttpMethod.GET));
    // Content-Type is not removed by the current handler implementation
    assertThat(outboundRequest.headers().contains("Content-Type"), is(true));
    assertThat(outboundRequest.headers().contains("Transfer-Encoding"), is(false));
    assertThat(outboundRequest.headers().get("Content-Length"), is(equalTo("0")));
  }

  @Test
  public void testRedirectWithSendBodyAlwaysMethodChangedButBodyNotSuppressed() {
    EmbeddedChannel channel = new EmbeddedChannel(new RedirectMethodChangeHandler());
    channel.attr(REDIRECT_CHANGE_METHOD).set(Boolean.TRUE);
    channel.attr(ALWAYS_SEND_BODY_KEY).set(Boolean.TRUE);

    HttpRequest request = new DefaultHttpRequest(HttpVersion.HTTP_1_1, HttpMethod.POST, "/test");
    request.headers().set("Content-Type", "application/json");
    request.headers().set("Content-Length", "100");

    channel.writeOutbound(request);
    HttpRequest outboundRequest = channel.readOutbound();
    assertThat(outboundRequest.method(), is(HttpMethod.GET));
    assertThat(outboundRequest.headers().contains("Content-Type"), is(true));
    assertThat(outboundRequest.headers().contains("Content-Length"), is(true));
  }
}
