/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.util;

/**
 * Utility class for encoding data into byte arrays.
 * <p>
 * This class provides methods for encoding unsigned integers in little-endian format. The class is designed for internal use and
 * offers low-level operations for data serialization.
 * </p>
 * <p>
 * This implementation is based on the jcifs library, available at <a href="https://github.com/kevinboone/jcifs">jcifs GitHub
 * Repository</a>.
 * </p>
 */
public class Encdec {

  /**
   * Private constructor to prevent instantiation of the utility class.
   */
  private Encdec() {}

  /**
   * Encodes a 32-bit unsigned integer in little-endian format.
   * <p>
   * The integer is split into four bytes, with the least significant byte stored at the lowest index of the destination array.
   * </p>
   *
   * @param i   The integer to encode.
   * @param dst The destination byte array where the encoded bytes will be written.
   * @param di  The starting index in the destination array where encoding will begin.
   */
  public static void enc_uint32le(int i, byte[] dst, int di) {
    dst[di++] = (byte) (i & 0xFF);
    dst[di++] = (byte) ((i >> 8) & 0xFF);
    dst[di++] = (byte) ((i >> 16) & 0xFF);
    dst[di] = (byte) ((i >> 24) & 0xFF);
  }

  /**
   * Encodes a 64-bit unsigned integer in little-endian format.
   * <p>
   * The long integer is split into two 32-bit parts, each encoded using the {@link #enc_uint32le(int, byte[], int)} method.
   * </p>
   *
   * @param l   The long integer to encode.
   * @param dst The destination byte array where the encoded bytes will be written.
   * @param di  The starting index in the destination array where encoding will begin.
   */
  public static void enc_uint64le(long l, byte[] dst, int di) {
    enc_uint32le((int) (l & 0xFFFFFFFFL), dst, di);
    enc_uint32le((int) ((l >> 32L) & 0xFFFFFFFFL), dst, di + 4);
  }
}
