/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.av;

import static org.mule.service.http.netty.impl.client.auth.ntlm.smb.SMBUtil.readInt8;
import static org.mule.service.http.netty.impl.client.auth.ntlm.smb.SMBUtil.writeInt8;

/**
 * Represents the AV_PAIR structure for a timestamp in NTLM authentication. This class encodes and decodes timestamps using an
 * 8-byte representation.
 *
 * <p>
 * This implementation is based on the jcifs library, available at:
 * <a href="https://github.com/codelibs/jcifs">https://github.com/codelibs/jcifs</a>
 * </p>
 *
 * @see <a href="https://github.com/codelibs/jcifs">jcifs on GitHub</a>
 */
public class AvTimestamp extends AvPair {

  /**
   * Constructs an instance of {@code AvTimestamp} with the specified raw byte data.
   *
   * @param raw the raw byte data representing the timestamp.
   */
  public AvTimestamp(byte[] raw) {
    super(MsvAvTimestamp, raw);
  }

  /**
   * Constructs an instance of {@code AvTimestamp} with the specified timestamp.
   *
   * @param ts the timestamp to encode.
   */
  public AvTimestamp(long ts) {
    this(encode(ts));
  }

  /**
   * Encodes a timestamp into an 8-byte array.
   *
   * @param ts the timestamp to encode.
   * @return a byte array containing the encoded timestamp.
   */
  private static byte[] encode(long ts) {
    byte[] data = new byte[8];
    writeInt8(ts, data, 0);
    return data;
  }

  /**
   * Retrieves the timestamp as a {@code long} value from the raw byte data.
   *
   * @return the decoded timestamp.
   */
  public long getTimestamp() {
    return readInt8(getRaw(), 0);
  }
}
