/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_LENGTH;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelOutboundHandlerAdapter;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.util.AttributeKey;

/**
 * This handler removes the content length if an attribute is set in the context of the request. We have to do this to preserve
 * backwards compatibility because there are some cases that we don't expect the content length to be set (when the method does
 * not require it, we are not forcing it with a sendBodyMode ALWAYS property and the body is empty) but Reactor-netty library
 * always sets the content length because the RFC does not mandate it to be sent (it has a SHOULD NOT)
 */
public class RemoveContentLengthHandler extends ChannelOutboundHandlerAdapter {

  @Override
  public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) {
    AttributeKey<Boolean> key = AttributeKey.valueOf("removeContentLength");
    Boolean removeContentLength = ctx.attr(key).get();
    if (removeContentLength != null && removeContentLength && msg instanceof HttpRequest httpRequest) {
      httpRequest.headers().remove(CONTENT_LENGTH);
    }
    ctx.write(msg, promise);
  }
}
