/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.integration;

import static org.mule.runtime.core.api.config.MuleProperties.APP_NAME_PROPERTY;
import static org.mule.runtime.core.api.config.MuleProperties.DOMAIN_NAME_PROPERTY;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.APP;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;

import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.MuleConfiguration;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpClientFactory;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.provider.NettyHttpServiceProvider;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistryBuilder;
import org.mule.service.http.netty.impl.service.NettyHttpServiceImplementation;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class NettyImplementationTestCase extends AbstractHttpTestCase {

  @Rule
  public DynamicPort dynamicPort = new DynamicPort("dynamicPort");

  private NettyHttpServiceImplementation httpService;

  @Before
  public void setUp() throws MuleException {
    SchedulerService schedulerService = mock(SchedulerService.class);
    NettyHttpServiceProvider httpServiceProvider = new NettyHttpServiceProvider();
    httpServiceProvider.setSchedulerService(schedulerService);

    httpService = (NettyHttpServiceImplementation) httpServiceProvider.getServiceDefinition().getService();
    startIfNeeded(httpService);
  }

  @After
  public void tearDown() throws MuleException {
    stopIfNeeded(httpService);
  }

  @Test
  public void createClient() {
    HttpClientConfiguration clientConfig = new HttpClientConfiguration.Builder()
        .setName("TheConfig")
        .build();

    HttpClientFactory clientFactory = httpService.getClientFactory();
    HttpClient httpClient = clientFactory.create(clientConfig);

    assertThat(httpClient, instanceOf(NettyHttpClient.class));
  }

  @Test
  public void createServerWithoutInjectedParams() throws ServerCreationException, ServerNotFoundException {
    testCreateServerWithFactory(httpService.getServerFactory());
  }

  @Test
  public void createServerWithInjectedParams() throws ServerCreationException, ServerNotFoundException {
    MuleConfiguration configuration = mock(MuleConfiguration.class);
    MuleContext muleContext = mock(MuleContext.class);
    when(muleContext.getConfiguration()).thenReturn(configuration);
    when(muleContext.getArtifactType()).thenReturn(APP);
    Registry registry = mock(Registry.class);
    when(registry.<String>lookupByName(APP_NAME_PROPERTY)).thenReturn(of("AppName"));
    when(registry.<String>lookupByName(DOMAIN_NAME_PROPERTY)).thenReturn(of("DomainName"));
    testCreateServerWithFactory(httpService.getServerFactory(muleContext, registry));
  }

  private void testCreateServerWithFactory(HttpServerFactory serverFactory)
      throws ServerCreationException, ServerNotFoundException {
    serverFactory.create(new HttpServerConfiguration.Builder()
        .setName("TheConfig")
        .setHost("localhost")
        .setPort(dynamicPort.getNumber())
        .build());
    assertThat(serverFactory.lookup("TheConfig").getClass().getSimpleName(), is("NoLifecycleHttpServer"));
  }

  @Test
  public void getRequestMatcher() {
    assertThat(httpService.getRequestMatcherRegistryBuilder(), instanceOf(DefaultRequestMatcherRegistryBuilder.class));
  }
}
