/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer.FLUSH_CONSOLIDATION_HANDLER_NAME;

import static io.netty.buffer.ByteBufUtil.getBytes;
import static io.netty.channel.ChannelFutureListener.CLOSE_ON_FAILURE;
import static io.netty.handler.codec.http.HttpResponseStatus.INTERNAL_SERVER_ERROR;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.service.http.common.server.sse.SseRequestHandler;
import org.mule.service.http.netty.impl.message.NettyHttpRequestAdapter;
import org.mule.service.http.netty.impl.server.util.DefaultServerAddress;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.streaming.BlockingBidirectionalStream;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.URISyntaxException;
import java.util.concurrent.Executor;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpContent;
import io.netty.handler.codec.http.HttpObject;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpResponse;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.handler.ssl.SslHandler;
import io.netty.util.ReferenceCountUtil;
import org.slf4j.Logger;

/**
 * Implementation of Netty inbound handler that adapts the object to the Mule HTTP API and forwards them to a
 * {@link HttpListenerRegistry}.
 * <p>
 * Note: When the Mule HTTP Service is used to implement a connector, this is the point where the request is forwarded to the flow
 * processors chain.
 */
public class ForwardingToListenerHandler extends SimpleChannelInboundHandler<HttpObject> {

  private static final Logger LOGGER = getLogger(ForwardingToListenerHandler.class);

  private final HttpListenerRegistry httpListenerRegistry;
  private final SslHandler sslHandler;
  private final Executor ioExecutor;

  private OutputStream currentRequestContentSink;

  public ForwardingToListenerHandler(HttpListenerRegistry httpListenerRegistry, SslHandler sslHandler,
                                     Executor ioExecutor) {
    this.httpListenerRegistry = httpListenerRegistry;
    this.sslHandler = sslHandler;
    this.ioExecutor = ioExecutor;

    this.currentRequestContentSink = null;
  }

  @Override
  public void channelRead0(ChannelHandlerContext ctx, HttpObject httpObject) throws Exception {
    if (httpObject instanceof HttpRequest httpRequest) {
      InetSocketAddress socketAddress = (InetSocketAddress) ctx.channel().localAddress();
      DefaultServerAddress serverAddress = new DefaultServerAddress(socketAddress.getAddress(), socketAddress.getPort());
      org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = nettyToMuleRequest(httpRequest, socketAddress);

      try {
        RequestHandler requestHandler = getRequestHandler(httpRequest, serverAddress, muleRequest);
        if (requestHandler instanceof SseRequestHandler && null != ctx.pipeline().get(FLUSH_CONSOLIDATION_HANDLER_NAME)) {
          // The FlushConsolidationHandler shows a considerable performance improvement
          // by not doing unnecessary flushes, but when SSE is enabled, we want all the
          // flushes to be executed immediately.
          ctx.pipeline().remove(FLUSH_CONSOLIDATION_HANDLER_NAME);
        }

        if (isFullRequest(httpObject)) {
          callHandlerSync(ctx, requestHandler, muleRequest);
        } else {
          callHandlerAsync(ctx, httpObject, requestHandler, muleRequest);
        }
      } catch (Exception exception) {
        if (exception instanceof RuntimeException && exception.getCause() instanceof URISyntaxException) {
          // if the URL is malformed we want to get the correct response
          handleMalformedUri(ctx, httpObject, ((URISyntaxException) exception.getCause()));
        } else {
          throw exception;
        }
      }
    }

    if (partialContentAvailable(httpObject)) {
      handleContent((HttpContent) httpObject);
    }
  }

  private void callHandlerAsync(ChannelHandlerContext ctx, HttpObject httpObject, RequestHandler requestHandler,
                                org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest) {
    ioExecutor.execute(() -> {
      try {
        callHandlerSync(ctx, requestHandler, muleRequest);
      } catch (Exception e) {
        LOGGER.atError()
            .setCause(e)
            .log("Error handling request to {}", muleRequest.getUri());
        handleRequestHandleError(ctx, httpObject);
      }
    });
  }

  private void callHandlerSync(ChannelHandlerContext ctx, RequestHandler requestHandler,
                               org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest) {
    requestHandler.handleRequest(new NettyHttpRequestContext(muleRequest, ctx, sslHandler),
                                 new NettyHttp1ResponseReadyCallback(ctx, muleRequest, ioExecutor));
  }

  private boolean isFullRequest(HttpObject httpObject) {
    return httpObject instanceof FullHttpRequest;
  }

  private boolean partialContentAvailable(HttpObject httpObject) {
    return httpObject instanceof HttpContent && !(httpObject instanceof FullHttpRequest);
  }

  private void handleContent(HttpContent content) throws IOException {
    // TODO W-19810580: Avoid the copy to byte[] and try to copy directly to the output stream
    // could we even avoid the copy completely by retaining the content and using it as the data source for the InputStream?
    // would it be worth it?
    byte[] frameData = getBytes(content.content());
    try {
      currentRequestContentSink.write(frameData, 0, frameData.length);
    } catch (IOException exception) {
      if ("Trying to write in a closed buffer".equals(exception.getMessage())) {
        LOGGER.info("Nobody is reading the payload, so we are ignoring part of the content...");
      } else {
        throw exception;
      }
    }

    if (content instanceof LastHttpContent) {
      currentRequestContentSink.close();
      currentRequestContentSink = null;
    }
  }

  private RequestHandler getRequestHandler(HttpRequest nettyRequest,
                                           DefaultServerAddress serverAddress,
                                           org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest) {
    if (nettyRequest.decoderResult().isFailure()) {
      RequestHandler requestHandler = httpListenerRegistry.getErrorHandler(nettyRequest.decoderResult().cause());
      if (requestHandler != null) {
        return requestHandler;
      }
    }
    return httpListenerRegistry.getRequestHandler(serverAddress, muleRequest);
  }

  @Override
  public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
    LOGGER.error("Exception caught", cause);
    ctx.close();
  }

  private org.mule.runtime.http.api.domain.message.request.HttpRequest nettyToMuleRequest(HttpRequest httpRequest,
                                                                                          InetSocketAddress localAddress) {
    return new NettyHttpRequestAdapter(httpRequest, localAddress, createContent(httpRequest));
  }

  private InputStream createContent(HttpRequest httpRequest) {
    if (httpRequest instanceof FullHttpRequest fullHttpRequest) {
      // TODO: W-19810580: See if we can avoid the copy and pass along a retained slice using ByteBufInputStream with
      // releaseOnClose
      return new ByteArrayInputStream(getBytes(fullHttpRequest.content()));
    } else {
      BlockingBidirectionalStream blockingBuffer = new BlockingBidirectionalStream();
      currentRequestContentSink = blockingBuffer.getOutputStream();
      return blockingBuffer.getInputStream();
    }
  }

  // Added for testing purposes.
  public SslHandler getSslHandler() {
    return sslHandler;
  }

  private static void handleMalformedUri(ChannelHandlerContext ctx, HttpObject httpObject, URISyntaxException exception) {
    HttpRequest httpRequest = (HttpRequest) httpObject;
    ByteBuf responseMsg = Unpooled.buffer();
    ByteBufUtil.writeUtf8(responseMsg,
                          String.format("HTTP request parsing failed with error: \"%s\"",
                                        exception.getMessage()));
    HttpResponse rejection =
        new DefaultFullHttpResponse(httpRequest.protocolVersion(), HttpResponseStatus.BAD_REQUEST, responseMsg);
    // TODO W-19810644: add coverage and check if this release is correct
    ReferenceCountUtil.release(httpObject);
    ctx.writeAndFlush(rejection).addListener(CLOSE_ON_FAILURE);
  }

  private static void handleRequestHandleError(ChannelHandlerContext ctx, HttpObject httpObject) {
    HttpRequest httpRequest = (HttpRequest) httpObject;
    HttpResponse rejection =
        new DefaultFullHttpResponse(httpRequest.protocolVersion(), INTERNAL_SERVER_ERROR);
    // TODO W-19810644: add coverage and check if this release is correct
    ReferenceCountUtil.release(httpObject);
    ctx.writeAndFlush(rejection).addListener(CLOSE_ON_FAILURE);
  }
}
