/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth;

import org.mule.runtime.http.api.client.auth.HttpAuthenticationType;

import java.net.URI;

import org.apache.commons.lang3.StringUtils;

/**
 * Constructs the digest auth headers depending on the different exchanges between client and server First the client will send an
 * AUTHORIZATION header with basic data about the realm and the server will respond with a wwwAuthenticateHeader to request more
 * data, once that data is returned the server will then proceed to authorize or not the client
 */
public class DigestAuthMessageFactory implements AuthHeaderFactory {

  // Status of the authentication, to avoid falling into bad states (as infinite recursions) if the server is not well
  // implemented.
  enum Status {

    // Initial state, no headers exchanged.
    NOT_STARTED,

    // In some auth methods there are multiple messages exchanged, for example 401 received,
    // we sent a message, and we're waiting for the challenge from the server
    WAITING_FOR_CHALLENGE,

    // Last message was responded and even if this one failed there is nothing to be done
    FINISHED,
  }

  private final String username;
  private final String password;
  private final Realm realm;
  private final URI uri;
  private final String methodName;
  private Status status;

  public DigestAuthMessageFactory(String username, String password, URI uri, String methodName) {
    this.username = username;
    this.password = password;
    this.realm = new Realm(username, password, uri);
    this.uri = uri;
    this.methodName = methodName;
    this.status = Status.NOT_STARTED;
  }

  @Override
  public boolean hasFinished() {
    return Status.FINISHED == status;
  }

  @Override
  public String getNextHeader(String wwwAuthenticateHeader) {
    switch (status) {
      case NOT_STARTED:
        // The server told us that we have to send a first challenge message to start authentication.
        status = Status.WAITING_FOR_CHALLENGE;
        return computeDigestHeader(wwwAuthenticateHeader);
      case WAITING_FOR_CHALLENGE:
        status = Status.FINISHED;
        if (wwwAuthenticateHeader != null) {
          // The server requests a second challenge message
          return computeDigestHeader(wwwAuthenticateHeader);
        } else {
          return null;
        }
      case FINISHED:
        return null;
      default:
        throw new IllegalStateException("Unknown state: " + status);
    }
  }

  // TODO: benchmark this method because of so many string concatenations
  private String computeDigestHeader(String wwwAuthHeader) {
    Realm reqRealm;
    if (StringUtils.isBlank(wwwAuthHeader)) {
      reqRealm = Realm.newRealm(realm, uri, methodName);
    } else {
      reqRealm = buildRealmFromAuthHeader(wwwAuthHeader, uri, methodName);
    }
    String realmUri = AuthUtils.computeRealmURI(uri, false, false);
    StringBuilder builder = new StringBuilder().append("Digest ");
    builder.append(String.format("username=\"%s\",", this.username));
    builder.append(String.format("realm=\"%s\",", reqRealm.getRealmName()));
    builder.append(String.format("nonce=\"%s\",", reqRealm.getNonce()));
    builder.append(String.format("uri=\"%s\",", realmUri));
    if (!StringUtils.isBlank(realm.getAlgorithm())) {
      builder.append(String.format("algorithm=\"%s\",", reqRealm.getAlgorithm()));
    }
    builder.append(String.format("response=\"%s\",", reqRealm.getResponse()));
    if (reqRealm.getOpaque() != null) {
      builder.append(String.format("opaque=\"%s\",", reqRealm.getOpaque()));
    }

    if (reqRealm.getQop() != null) {
      builder.append(String.format("qop=\"%s\",", reqRealm.getQop()));
      // nc and cnonce only sent if server sent qop
      builder.append(String.format("nc=\"%s\",", reqRealm.getNc()));
      builder.append(String.format("cnonce=\"%s\",", reqRealm.getCnonce()));
    }

    String digestStringWithTrailingComma = builder.toString();
    return digestStringWithTrailingComma.substring(0, digestStringWithTrailingComma.length() - 2);
  }

  public Realm buildRealmFromAuthHeader(String headerLine, URI uri, String methodName) {
    String realmName = getTokenValue(headerLine, "realm");
    String nonce = getTokenValue(headerLine, "nonce");
    String opaque = getTokenValue(headerLine, "opaque");
    String scheme = StringUtils.isBlank(nonce) ? HttpAuthenticationType.BASIC.name() : HttpAuthenticationType.DIGEST.name();
    String algorithm = null;
    if (!StringUtils.isBlank(nonce)) {
      algorithm = getTokenValue(headerLine, "algorithm");
    }

    String rawQop = getTokenValue(headerLine, "qop");
    String qop = rawQop;
    if (rawQop != null) {
      qop = parseQop(rawQop);
    }

    return Realm.newRealm(this.realm, realmName, nonce, qop, opaque, algorithm, uri, methodName);
  }

  private static String getTokenValue(String headerLine, String token) {
    if (headerLine == null) {
      return null;
    }

    int match = headerLine.indexOf(token);
    if (match <= 0) {
      return null;
    }

    // = to skip
    match += token.length() + 1;
    int trailingComa = headerLine.indexOf(',', match);
    String value = headerLine.substring(match, trailingComa > 0 ? trailingComa : headerLine.length());
    value = value.length() > 0 && value.charAt(value.length() - 1) == '"'
        ? value.substring(0, value.length() - 1)
        : value;
    return value.charAt(0) == '"' ? value.substring(1) : value;
  }

  private static String parseQop(String rawQop) {
    String[] rawServerSupportedQops = rawQop.split(",");
    String[] serverSupportedQops = new String[rawServerSupportedQops.length];
    for (int i = 0; i < rawServerSupportedQops.length; i++) {
      serverSupportedQops[i] = rawServerSupportedQops[i].trim();
    }

    // auth over auth-int
    for (String qop : serverSupportedQops) {
      if ("auth".equals(qop)) {
        return qop;
      }
    }

    for (String qop : serverSupportedQops) {
      if ("auth-int".equals(qop)) {
        return qop;
      }
    }

    return null;
  }
}
