/*
 * (C) 2006 SAP XI 7.1 Adapter Framework Resource Adapter Skeleton
 */

package com.mulesoft.adapter.ra;

import java.io.PrintWriter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;

import javax.naming.InitialContext;
import javax.resource.ResourceException;
import javax.resource.spi.ConnectionEvent;
import javax.resource.spi.ConnectionManager;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionFactory;
import javax.resource.spi.security.PasswordCredential;
import javax.security.auth.Subject;

import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.PILogger;
import com.mulesoft.adapter.helper.TopicListnenerLocking;
import com.mulesoft.adapter.module.salesforce.SalesforcePIModule;
import com.mulesoft.adapter.ra.ChannelOperationTuple.Operation;
import com.sap.aii.af.service.administration.api.cpa.CPAFactory;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.aii.af.service.cpa.CPAObjectType;
import com.sap.aii.af.service.cpa.Channel;
import com.sap.aii.af.service.cpa.Direction;
import com.sap.aii.af.service.resource.SAPAdapterResources;
import com.sap.engine.interfaces.connector.ManagedConnectionFactoryActivation;
import com.sap.engine.interfaces.messaging.api.PublicAPIAccessFactory;
import com.sap.engine.interfaces.messaging.api.auditlog.AuditAccess;
import com.sap.guid.GUID;

/**
 * An object of the class <code>SpiManagedConnectionFactory</code> (MCF) is a
 * factory of both, <code>ManagedConnection</code> and
 * <code>CciConnectionFactory</code> instances. This class supports connection
 * pooling by defining methods for matching and creating connections.
 * <code>SpiManagedConnectionFactory</code> MUST be a Java Bean to be JCA
 * compliant (see pg. 51) A SAP J2EE server peculiarity is the implementation of
 * the <code>ManagedConnectionFactoryActivation</code> interface. This allows
 * the MCF to be informed when the adapter is being started or stopped. It
 * represents a JCA 1.0 workaround for the missing JCA 1.5 activation feature in
 * NetWeaver'04 (hence it is deprecated since the next major NetWeaver version
 * supports JCA 1.5).
 * 
 * <b>Changes from 3.0/7.0 to 7.1:</b> 1. Repackaging
 * com.sap.engine.interfaces.messaging 2. AuditLog simplified: MessageDirection,
 * MessageKey instead of AuditDirection, AuditKey 3. MessageIDMapper: 4. Use of
 * SAP J2EE transaction manager instead of JTA. (com.sap.transaction.TxManager
 * instead of javax.transaction.TransactionManager) 5. XIInteractionSpec: START,
 * STOP removed, is offered by com.sap.aii.af.service.administration.api now 6.
 * XIMessageRecord does not extend
 * com.sap.engine.interfaces.messaging.api.Message anymore, but wraps it now.
 * Use get/setMessage() instead of direct calls. 7. Introduce XITrace utility
 * class and refer to J2EE logging directly 8. Built against the new XI AF
 * facades, see cross reference Excel for details. 9. ModuleProcessor reference
 * is not cached. (7.1 SP3 change)
 * 
 * @version: $Id:
 *           //tc/xpi.external/NW730EXT_01_REL/src/_sample_rar_module/rar/src
 *           /com
 *           /sap/aii/af/sample/adapter/ra/SPIManagedConnectionFactory.java#1 $
 **/

public class SPIManagedConnectionFactory implements ManagedConnectionFactory, Serializable, Runnable, ManagedConnectionFactoryActivation {

    // See Serializable
    static final long serialVersionUID = -2387046407149571208L;

    // Trace and audit log
    private static final XITrace TRACE = new XITrace(SPIManagedConnectionFactory.class.getName());
    private AuditAccess audit = null;

    // For debug purposes: Use a GUID and a timer for "keep alive" trace
    // messages
    private GUID mcfLocalGuid = null; // GUID of this mcf

    // Some internal control parameters
    private static int waitTime = 5000;

    public static final String JNDI_NAME = "deployedAdapters/com.mulesoft.adapter.ra/shareable/com.mulesoft.adapter.ra";

    // The JCA logwriter stream (not used by the XI AF)
    transient PrintWriter logWriter;

    // An object for synchronization purposes
    private static Object synchronizer = new Object();

    // Use the XI AF SAPAdapterResources for thread and transaction management
    // as replacement for missing JCA 1.5 functionality
    private SAPAdapterResources msRes = null;

    // Thread status of the inbound processing thread
    private int threadStatus = TH_INIT;
    private static final int TH_INIT = 0;
    private static final int TH_STARTED = 1;
    private static final int TH_STOPPED = 2;

    // The initial context for JNDI lookups
    private InitialContext ctx = null;

    // A specific class that manages the XI configuration
    private XIConfiguration xIConfiguration = null;

    // A Map to manage all created ManagedConnections
    // Although the J2EE JCA container manages pools of ManagedConnections as
    // well, it is reasonable
    // to maintain the own ManagedConnection objects because ManagedConnection
    // in this sample
    // has a 1:1 relationship to XI receiver channels. And those XI receiver
    // channles might be removed, changed, etc.
    // by XI CPA cache updates which are not synchronized with the JCA
    // container.
    // => When the ManagedConnection is removed (by XI CPA) it must be
    // invalidated here and the container must be informed (via event
    // connection.closed)
    // The Map must be synchronized because it might be changed by a CPA update
    // or the JCA container in parallel.
    private Map<String, SPIManagedConnection> managedConnections = Collections.synchronizedMap(new HashMap<String, SPIManagedConnection>());

    // XI AF Message ID mapper
    private transient XIMessageFactoryImpl mf = null;

    // Address modes
    private static final String AM_CPA = "CPA";

    // The next MCF properties are typical JCA MCF settings that can't be
    // set in the XI ID since the values must be known beforehand
    // Hence the adapter waits during the startup phase till all values are
    // known
    private String addressMode = null;
    private String adapterType = null;
    private String adapterNamespace = null;
    private int propWaitNum = 10;
    private int propWaitTime = 1000;

    private final Map<String, SalesforcePIModule> modulesPerChannel = new ConcurrentHashMap<String, SalesforcePIModule>();
    private final BlockingQueue<ChannelOperationTuple> channelUpdateQueue = new LinkedBlockingQueue<ChannelOperationTuple>();

    // Adapter specific message attributes (ASMA)
    static final String ASMA_NAME = "JCAChannelID";

    /**
     * The JCA specification requires a default constructor since this class
     * must be Java Bean compliant and is propagated from this class to them. XI
     * specific is the access to the SAPAdapterResource typed objects that allow
     * access to the SAP J2EE thread and transaction manager as substitute for
     * the missing work and inbound transaction concepts of JCA 1.0. (SPI JCA
     * 1.0)
     * 
     * @throws ResourceException
     *             not thrown
     */
    public SPIManagedConnectionFactory() throws ResourceException {
        final String SIGNATURE = "SpiManagedConnectionFactory()";
        TRACE.entering(SIGNATURE);

        // Examine this method to see all possibilities the SAP J2EE logging API
        // offers

        // Access the XI AF resources
        // CS_THREADMGR START
        try {
            ctx = new InitialContext();
            Object res = ctx.lookup("SAPAdapterResources");
            msRes = (SAPAdapterResources) res;
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0011", "Access to XI AF MS resource failed. Adapter cannot be started.");
        }
        // CS_THREADMGR END

        // Optional: Set mcf GUID for debugging purposes
        try {
            synchronized (synchronizer) {
                // CS_GUID START
                mcfLocalGuid = new GUID();
                TRACE.infoT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "This SPIManagedConnectionFactory has the GUID: " + mcfLocalGuid.toString());
                // CS_GUID END
            }
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF,
                    "Creation of MCF GUID failed. Thus no periodic status report possible! Reason: " + e.getMessage());
        }

        // Important: DO NOT start the inbound processing here! Instead start it
        // when start() is being called!
        // The inbound processing must be stopped when stop() is called
        // If this rule is not considered the setting of the MCF properties via
        // the J2EE visual admin will end
        // in two MCF instances running both the inbound processing. Usually
        // conflicts with non-shareable ressources will happen then.

        TRACE.exiting(SIGNATURE);
    }

    private void addTopicSubscription(Channel channel) {
        IPILogger logger = new PILogger(channel, null);
        try {
            TopicListnenerLocking locking = new TopicListnenerLocking(channel);

            if (locking.aquireLock()) {
                final SalesforcePIModule pIModule = new SalesforcePIModule(channel);
                SPIManagedConnectionFactory.this.modulesPerChannel.put(channel.getObjectId(), pIModule);
                pIModule.subscribe(SPIManagedConnectionFactory.this.mf, logger);
            } else {
                logger.reportProcessingStatus(ProcessState.OK, "Topic already locked, please see other cluster nodes of this channel");
            }
        } catch (Exception e) {
            final String SIGNATURE = "addTopicSubscription";
            TRACE.catching(SIGNATURE, e);
            logger.reportProcessingStatus(ProcessState.FATAL, e.getMessage());
        }
    }

    private void removeTopicSubscription(Channel channel) {
        IPILogger logger = new PILogger(channel, null);
        final SalesforcePIModule piModule = SPIManagedConnectionFactory.this.modulesPerChannel.remove(channel.getObjectId());

        if (piModule != null) {
            TopicListnenerLocking locking = new TopicListnenerLocking(channel);

            try {
                locking.unlock();
                piModule.unsubscribe(logger);
            } catch (Exception e) {
                final String SIGNATURE = "removeTopicSubscription";
                TRACE.catching(SIGNATURE, e);
                logger.reportProcessingStatus(ProcessState.FATAL, e.getMessage());
            }
        }
    }

    /**
     * This factory method is used in the context of XI AF: The J2EE server
     * passed a <code>ConnectionManager</code> object via the
     * <code>createConnectionFactory</code> call to the
     * <code>ManagedConnectionFactory</code>, which in turn will the approriate
     * constructor of the <code>CciConnectionFactory</code>. (SPI JCA 1.0)
     * 
     * @param cm
     *            <code>ConnectionManager</code> of the J2EE server
     **/
    public Object createConnectionFactory(ConnectionManager cm) throws ResourceException {
        final String SIGNATURE = "createConnectionFactory(ConnectionManager cxManager)";
        TRACE.entering(SIGNATURE, new Object[] { cm });
        CCIConnectionFactory factory = new CCIConnectionFactory(this, cm);
        TRACE.exiting(SIGNATURE);
        return factory;
    }

    /**
     * This factory method is used for the two-tier, non J2EE container
     * approach: The <code>ConnectionManager</code> implementation of the
     * resource adapter is used instead of the J2EE
     * <code>ConnectionManager</code>. (SPI JCA 1.0)
     **/
    public Object createConnectionFactory() throws ResourceException {
        final String SIGNATURE = "createConnectionFactory()";
        TRACE.entering(SIGNATURE);
        CCIConnectionFactory factory = new CCIConnectionFactory(this, null);
        TRACE.exiting(SIGNATURE);
        return factory;
    }

    /**
     * This factory method is used called by the J2EE server to create a new
     * managed connection. Note that it does not necessarily mean that a new
     * physical connection is being created. However, in this sample a new file
     * is opened. (SPI JCA 1.0)
     * 
     * @param subject
     *            JAAS authentification data with logon credentials to open the
     *            physical connection
     * @param info
     *            <code>ConnectionRequestInfo</code> with additional information
     *            to open the managed connection
     * @return New created managed connection of class
     *         <code>SpiManagedConnection</code>
     * @throws ResourceException
     *             Thrown if managed connection cannot be created
     **/
    public ManagedConnection createManagedConnection(Subject subject, ConnectionRequestInfo info) throws ResourceException {
        final String SIGNATURE = "createManagedConnection(Subject subject, ConnectionRequestInfo info)";
        TRACE.entering(SIGNATURE, new Object[] { subject, info });
        String channelID = null;
        Channel channel = null;

        // Check input
        SPIManagedConnection mc = null;

        if (!(info instanceof CCIConnectionRequestInfo)) {
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0013",
                    "Received an unknown ConnectionRequestInfo. Cannot determine channelId!");
            ResourceException re = new ResourceException("Received an unknown ConnectionRequestInfo. Cannot determine channelId!");
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }

        // CS_CIDLKUP START
        // Determine channel configuration. In this sample it is just the output
        // directory and file name
        try {
            channelID = ((CCIConnectionRequestInfo) info).getChannelId();
            channel = CPAFactory.getInstance().getLookupManager().getCPAObject(CPAObjectType.CHANNEL, channelID);
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0014", "Cannot access the channel parameters of channel: " + channelID
                    + ". Check whether the channel is stopped in the administrator console.");
            ResourceException re = new ResourceException("Cannot access the channel parameters of channel: " + channelID
                    + ". Check whether the channel is stopped in the administrator console.");
            throw re;
        }
        // CS_CIDLKUP END

        PasswordCredential credential = XISecurityUtilities.getPasswordCredential(this, subject, info);
        mc = new SPIManagedConnection(this, credential, false, channelID, channel);

        // Store the new mc in the map
        if (mc != null) {
            managedConnections.put(channelID, mc);
            TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "For channelID {0} this managed connection is stored: {1}", new Object[] { channelID, mc });
        }

        TRACE.exiting(SIGNATURE);
        return mc;
    }

    /**
     * 
     * When the XI CPA Cache triggers a channel remove it makes sense to remove
     * the associated ManagedConnection as well. Advantages: - Early release of
     * resources (physical connections, JCA connections, ...) - JCA pool cleanup
     * (ra implementation specific)
     * 
     * @param channelID
     *            XI channel ID that identifies the ManagedConnection that has
     *            to be removed
     * @throws ResourceException
     *             Thrown if managed connection cannot be created
     **/
    void destroyManagedConnection(String channelID) throws ResourceException {
        final String SIGNATURE = "destroyManagedConnection(String channelID)";
        TRACE.entering(SIGNATURE, new Object[] { channelID });
        SPIManagedConnection mc = null;
        try {
            // Lookup the MC that is related to the XI channel
            mc = (SPIManagedConnection) managedConnections.get(channelID);
            if (mc != null) {
                // Inform all SPI event listeners (i.e. the JCA container that
                // this mc is destroyed now)
                mc.sendEvent(ConnectionEvent.CONNECTION_CLOSED, null, mc);
                managedConnections.remove(channelID);
                mc.destroy(true);
                TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "ManagedConnection for channel ID {0} found and destroyed.", new Object[] { channelID });
            } else
                TRACE.warningT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "ManagedConnection for channel ID {0} not found.", new Object[] { channelID });
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0015",
                    "Received exception during ManagedConnection destroy: " + e.getMessage());
        }
        TRACE.exiting(SIGNATURE);
    }

    /**
     * 
     * If a ManagedConnection is destroyed by the JCA container it reports this
     * to its ManagedConnectionFactory. The MCF then removes the MC from the
     * internal pool. (ra implementation specific)
     * 
     * @param channelID
     *            XI channel ID that identifies the ManagedConnection that has
     *            to be removed
     **/
    void removeManagedConnection(String channelID) {
        final String SIGNATURE = "removeManagedConnection(String channelID)";
        TRACE.entering(SIGNATURE, new Object[] { channelID });
        managedConnections.remove(channelID);
        TRACE.exiting(SIGNATURE);
    }

    /**
     * This method is not a factory method but more a selection method: It
     * selects the best-fitting managed connection contained in the connection
     * set based on the subject (user, password) and connection information (
     * <code>ConnectionRequestInfo</code>). This sample simply checks the
     * equality of the subject and that the managed connection was opened for a
     * given XI channel. (SPI JCA 1.0)
     * 
     * @param connectionSet
     *            containing <code>SpiManagedConnection</code> objects
     * @param subject
     *            JAAS authentification data with logon credentials to open the
     *            physical connection
     * @param info
     *            <code>ConnectionRequestInfo</code> with additional information
     *            to open the managed connection
     * @return Existing managed connection of class
     *         <code>SpiManagedConnection</code>
     * @throws ResourceException
     *             Thrown if managed connection cannot be created
     **/
    public ManagedConnection matchManagedConnections(@SuppressWarnings("rawtypes") Set connectionSet, Subject subject, ConnectionRequestInfo info)
            throws ResourceException {
        final String SIGNATURE = "matchManagedConnections(Set connectionSet, Subject subject, ConnectionRequestInfo info)";
        TRACE.entering(SIGNATURE, new Object[] { connectionSet, subject, info });

        SPIManagedConnection mcFound = null;
        CCIConnectionRequestInfo cciInfo = null;
        PasswordCredential pc = XISecurityUtilities.getPasswordCredential(this, subject, info);

        if (info instanceof CCIConnectionRequestInfo)
            cciInfo = (CCIConnectionRequestInfo) info;
        else {
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "Unknown ConnectionRequestInfo parameter received. Cannot match connection");
            return null;
        }

        Iterator<?> it = connectionSet.iterator();

        // A managed connection is returned if the channelID which is requested
        // is the same
        while (it.hasNext() && (mcFound == null)) {
            Object obj = it.next();
            if (obj instanceof SPIManagedConnection) {
                SPIManagedConnection mc = (SPIManagedConnection) obj;
                if (!mc.isDestroyed()) {
                    ManagedConnectionFactory mcf = mc.getManagedConnectionFactory();
                    if ((XISecurityUtilities.isPasswordCredentialEqual(mc.getPasswordCredential(), pc)) && (mcf.equals(this))
                            && (mc.getChannelID().equalsIgnoreCase(cciInfo.getChannelId()))) {
                        mcFound = mc;
                        TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT, "Found existing ManagedConnection in container set for channel {0}.",
                                new Object[] { mc.getChannelID() });
                    } else
                        TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT, "ManagedConnection in container set does not fit. Ignore.");
                } else
                    TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT, "Destroyed sample ManagedConnection in container set. Ignore.");
            } else
                TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT, "This is not a sample ManagedConnection in container set. Ignore.");
        }
        TRACE.exiting(SIGNATURE);
        return mcFound;
    }

    /**
     * Sets the JCA J2EE logwriter <code>PrintWriter</code> object. Although JCA
     * specifies this mechansim it is not being used by XI AF. Instead the
     * resource adapters should use the XI AF trace service classes as done here
     * in this sample.
     * 
     * (SPI JCA 1.0)
     * 
     * @param out
     *            <code>PrintWriter</code> print writer for logging purposes
     */
    public void setLogWriter(PrintWriter out) throws ResourceException {
        final String SIGNATURE = "setLogWriter(PrintWriter out)";
        TRACE.entering(SIGNATURE, new Object[] { out });
        out.print("XI AF Sample Adapter has received a J2EE container log writer.");
        out.print("XI AF Sample Adapter will not use the J2EE container log writer. See the trace file for details.");
        logWriter = out;
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Gets the JCA J2EE logwriter <code>PrintWriter</code> object. (SPI JCA
     * 1.0)
     * 
     * @return <code>PrintWriter</code> print writer for logging purposes
     */
    public PrintWriter getLogWriter() throws ResourceException {
        return logWriter;
    }

    /**
     * Gets the XI AF audit log access <code>AuditAccess</code> object. (ra
     * implementation specific)
     * 
     * @return <code>AuditAccess</code> for logging purposes, might be null
     *         during the start up phase
     */
    AuditAccess getAuditAccess() {
        return audit;
    }

    /**
     * Gets the XI AF message factory wrapper <code>XIMessageFactoryImpl</code>
     * object. (ra implementation specific)
     * 
     * @return <code>XIMessageFactoryImpl</code> to create messages
     */
    XIMessageFactoryImpl getXIMessageFactoryImpl() {
        return mf;
    }

    /**
     * A JCA 1.0 compliant <code>SpiManagedConnectionFactory</code> must
     * implement the <code>equals()</code> to allow the J2EE container a
     * sensible connection pooling. The equality MUST be defined on the complete
     * property set (see pg. 50)
     * 
     * (SPI JCA 1.0)
     * 
     * @param obj
     *            Object to compare
     * @return True of obj is equal (but not necessarily identical) with this
     *         <code>SpiManagedConnectionFactory</code>, false otherwise
     */
    @Override
    public boolean equals(Object obj) {
        final String SIGNATURE = "equals(Object obj)";
        TRACE.entering(SIGNATURE, new Object[] { obj });
        boolean equal = false;
        if (obj instanceof SPIManagedConnectionFactory) {
            SPIManagedConnectionFactory other = (SPIManagedConnectionFactory) obj;
            if ((adapterNamespace.equals(other.getAdapterNamespace())) && (adapterType.equals(other.getAdapterType()))
                    && (addressMode.equals(other.getAddressMode())))
                equal = true;
        }
        TRACE.exiting(SIGNATURE);
        return equal;
    }

    /**
     * A JCA 1.0 compliant <code>SpiManagedConnectionFactory</code> must
     * implement the <code>hashCode()</code> to allow the J2EE container a
     * sensible connection pooling. The equality MUST be defined on the complete
     * property set (see pg. 50)
     * 
     * (SPI JCA 1.0)
     * 
     * @return Integer value representing the hash code of this
     *         <code>SpiManagedConnectionFactory</code>
     */
    @Override
    public int hashCode() {
        final String SIGNATURE = "hashCode()";
        TRACE.entering(SIGNATURE);
        int hash = 0;
        String propset = adapterNamespace + adapterType + addressMode;
        hash = propset.hashCode();
        TRACE.exiting(SIGNATURE);
        return hash;
    }

    /**
     * This class must be JavaBean compliant hence it offers getters for
     * properties
     * 
     * @return The address determination mode
     */
    public String getAddressMode() {
        final String SIGNATURE = "getAddressMode()";
        TRACE.entering(SIGNATURE);
        TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT, "Address determination mode =" + addressMode);
        TRACE.exiting(SIGNATURE);
        return addressMode;
    }

    /**
     * This class must be JavaBean compliant hence it offers setters for
     * properties
     * 
     * @param addressMode
     *            The address determination mode
     */
    public void setAddressMode(String addressMode) {
        this.addressMode = addressMode;
    }

    /**
     * Starts the inbound processing (ra implementation specific)
     * 
     * @throws ResourceException
     *             Thrown if thread cannot be started
     */
    public void startMCF() throws ResourceException {
        final String SIGNATURE = "startMCF()";
        TRACE.entering(SIGNATURE);

        // CS_THSTR START
        if (threadStatus != TH_STARTED) {

            /*
             * XI AF ra's MUST NOT use Java native threads Thread
             * inboundSimulator = null; try { inboundSimulator = new
             * Thread(null, this, "XI AF Sample Adapter Inbound"); }
             * catch(Exception e) { TRACE.errorT(SIGNATURE,
             * XIAdapterCategories.CONNECT_AF,
             * "Cannot create inbound message thread"); ResourceException re =
             * new ResourceException(e.getMessage()); TRACE.throwing(SIGNATURE,
             * re); throw re; }
             */

            try {
                threadStatus = TH_STARTED;
                msRes.startRunnable(this);
                // inboundSimulator.start(); see above
            } catch (Exception e) {
                TRACE.catching(SIGNATURE, e);
                threadStatus = TH_STOPPED;
                TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0016", "Cannot start inbound message thread");
                ResourceException re = new ResourceException(e.getMessage());
                TRACE.throwing(SIGNATURE, re);
                throw re;
            }
            // CS_THSTR END

        }
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Stops the inbound processing (ra implementation specific)
     */
    public void stopMCF() throws ResourceException {
        final String SIGNATURE = "stopMCF()";
        TRACE.entering(SIGNATURE);

        unregisterSubscriptions();

        // Inform run thread
        threadStatus = TH_STOPPED;
        try {
            xIConfiguration.stop();
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0017", "Cannot stop inbound message thread. Reason: " + e.getMessage());
            ResourceException re = new ResourceException(e.getMessage());
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }
        TRACE.exiting(SIGNATURE);
    }

    private void unregisterSubscriptions() throws ResourceException {
        // unregister all subscribed modules
        LinkedList<Channel> inboundChannels = xIConfiguration.getCopy(Direction.INBOUND);
        List<ChannelOperationTuple> channelOperationTuples = new ArrayList<ChannelOperationTuple>(inboundChannels.size());
        for (Channel inboundChannel : inboundChannels) {
            ChannelOperationTuple channelOperationTuple = new ChannelOperationTuple(inboundChannel, Operation.Remove);
            channelOperationTuples.add(channelOperationTuple);
        }
        updateChannels(channelOperationTuples);
        
        do {
            try {
                Thread.sleep(waitTime + 1000);
            } catch (InterruptedException e1) {
                Thread.currentThread().interrupt();
            }
        } while (!modulesPerChannel.isEmpty());
    }

    /**
     * Simulates incoming messages in a separate thread. (ra implementation
     * specific)
     */
    public void run() {
        final String SIGNATURE = "run()";
        TRACE.entering(SIGNATURE);

        // CS_MCFTNAMESET START
        String oldThreadName = Thread.currentThread().getName();
        String newThreadName = "XI AF Mule MCF " + mcfLocalGuid;
        try {
            Thread.currentThread().setName(newThreadName);
            TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "Switched thread name to: {0}", new Object[] { newThreadName });
            // CS_MCFTNAMESET END

            // CS_MCFPROPS START
            // Wait now till the MCF properties are set that cannot be
            // configured via XI ID
            boolean notSet = true;
            int numTry = 0;

            while ((notSet) && (numTry < propWaitNum)) {
                if ((addressMode != null) && (adapterType != null) && (adapterNamespace != null))
                    notSet = false;
                numTry++;
                TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "MCF waits for setter completion. Try: {0} of {1}.",
                        new Object[] { Integer.toString(numTry), Integer.toString(propWaitNum) });
                try {
                    Thread.sleep(propWaitTime);
                } catch (Exception e) {
                    TRACE.catching(SIGNATURE, e);
                }
                ;
            }
            if (addressMode == null)
                addressMode = AM_CPA;
            if (adapterType == null)
                adapterType = "SFDC_GATEWAY";
            if (adapterNamespace == null)
                adapterNamespace = "urn:mulesoft.com:pi:sfdcgateway:meta";
            // CS_MCFPROPS END

            // Create the helper object that manages the XI channel information
            if (xIConfiguration == null) {
                try {
                    xIConfiguration = new XIConfiguration(adapterType, adapterNamespace);
                    xIConfiguration.init(this);
                } catch (Exception e) {
                    TRACE.catching(SIGNATURE, e);
                    TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0018",
                            "Cannot instatiate the XI CPA handler. The inbound processing is stopped. Exception:" + e.toString());
                    threadStatus = TH_STOPPED;
                }
            }

            while (threadStatus == TH_STARTED) {

                try {
                    while (!channelUpdateQueue.isEmpty()) {
                        ChannelOperationTuple channelOperationTuple = channelUpdateQueue.poll();
                        if (channelOperationTuple.getOperation() == Operation.Remove) {
                            removeTopicSubscription(channelOperationTuple.getChannel());
                        } else {
                            addTopicSubscription(channelOperationTuple.getChannel());
                        }

                    }
                } catch (Exception e) {
                    TRACE.catching(SIGNATURE, e);
                    TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0019",
                            "Cannot access inbound channel configuration. Received exception: " + e.getMessage());
                }

                try {
                    synchronized (this) {
                        wait(waitTime);
                    }
                } catch (InterruptedException e1) {
                    TRACE.catching(SIGNATURE, e1);
                    TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0020",
                            "Inbound thread stopped. Received exception during wait period: " + e1.getMessage());
                    threadStatus = TH_STOPPED;
                }
            }

        } finally {
            Thread.currentThread().setName(oldThreadName);
            TRACE.debugT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "Switched thread name back to: {0}", new Object[] { oldThreadName });
            // CS_MCFTNAMERESET END
        }
    }

    public void addChannel(Channel channel) {
        channelUpdateQueue.add(new ChannelOperationTuple(channel, Operation.Add));
    }

    public void removeChannel(Channel channel) {
        channelUpdateQueue.add(new ChannelOperationTuple(channel, Operation.Remove));
    }

    public void updateChannels(List<ChannelOperationTuple> channelOperations) {
        channelUpdateQueue.addAll(channelOperations);
    }

    /**
     * Getter for the adapterNamespace for JCA ra configuration. The
     * adapterNamespace must be equal to related one configured in the XI IR.
     * 
     * @return String current adapterNamespace
     */
    public String getAdapterNamespace() {
        final String SIGNATURE = "getAdapterNamespace()";
        TRACE.entering(SIGNATURE);
        TRACE.exiting(SIGNATURE);
        return adapterNamespace;
    }

    /**
     * Getter for the adapterType for JCA ra configuration. The adapterType must
     * be equal to related one configured in the XI IR.
     * 
     * @return String current adapterType
     */
    public String getAdapterType() {
        return adapterType;
    }

    /**
     * Setter for the adapterNamespace for JCA ra configuration. The
     * adapterNamespace must be equal to related one configured in the XI IR.
     * 
     * @param adapterNamespace
     *            for XI CPA lookup
     */
    public void setAdapterNamespace(String adapterNamespace) {
        final String SIGNATURE = "setAdapterNamespace(String adapterNamespace)";
        TRACE.entering(SIGNATURE, new Object[] { adapterNamespace });
        this.adapterNamespace = adapterNamespace;
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Setter for the adapterType for JCA ra configuration. The adapterType must
     * be equal to related one configured in the XI IR.
     * 
     * @param adapterType
     *            for XI CPA lookup
     */
    public void setAdapterType(String adapterType) {
        final String SIGNATURE = "setAdapterType(String adapterType)";
        TRACE.entering(SIGNATURE, new Object[] { adapterType });
        this.adapterType = adapterType;
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Getter for the MCF GUID
     * 
     * @return GUID Guid of this MCF (ra implementation specific)
     */
    public GUID getMcfLocalGuid() {
        return mcfLocalGuid;
    }

    /**
     * The <code>start()</code> is called by the J2EE JCA container when the JCA
     * adapter is started via the J2EE admin console (e.g. explicitly in the
     * connector service or implicitly during a deployment). The inbound
     * processing MUST be started now.
     * 
     * @see com.sap.engine.interfaces.connector.ManagedConnectionFactoryActivation#start()
     *      (ra implementation specific)
     **/
    // CS_MCFASTART START
    public void start() {
        final String SIGNATURE = "start()";
        TRACE.entering(SIGNATURE);
        String controlledMcfGuid = this.getMcfLocalGuid().toHexString();
        TRACE.infoT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "MCF with GUID {0} is started now. ({1})", new Object[] { controlledMcfGuid.toString(),
                SPIManagedConnectionFactory.class.getClassLoader() });

        // Please note: The sample does not contain a status of the MCF. You
        // should add status like init, start, stop, starting and stopping
        // to control the start() and stop() calls coming from the J2EE JCA
        // container

        // Get access to the audit log
        // CS_AUDITACCESS START
        try {
            audit = PublicAPIAccessFactory.getPublicAPIAccess().getAuditAccess();
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT, "SOA.apt_sample.0035",
                    "Unable to access the XI AF audit log. Reason: {0}. Adapter cannot not start the inbound processing!", new Object[] { e });
            TRACE.exiting(SIGNATURE);
            return;
        }
        // CS_AUDITACCESS START

        // Create a message factory for XI messages
        try {
            mf = new XIMessageFactoryImpl(adapterType, adapterNamespace);
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT, "SOA.apt_sample.0037",
                    "Unable to create XI message factory. Adapter cannot not start the inbound processing!");
            TRACE.exiting(SIGNATURE);
            return;
        }

        // start the inbound processing
        try {
            this.startMCF();
            TRACE.infoT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "MCF with GUID {0} was started successfully.", new Object[] { controlledMcfGuid.toString() });
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "SOA.apt_sample.0038", "Start of MCF failed. Reason: {0}", new Object[] { e.getMessage() });
        }
    }

    // CS_MCFASTART END

    /**
     * The <code>stop()</code> is called by the J2EE JCA container when the JCA
     * adapter is stopped via the J2EE admin console (e.g. explicitly in the
     * connector service or implicitly during a deployment). The inbound
     * processing MUST be stopped now, shared ressources such as ports MUST be
     * released now.
     * 
     * @see com.sap.engine.interfaces.connector.ManagedConnectionFactoryActivation#stop()
     *      (ra implementation specific)
     **/
    // CS_MCFASTOP START
    public void stop() {
        final String SIGNATURE = "stop()";
        TRACE.entering(SIGNATURE);
        String controlledMcfGuid = this.getMcfLocalGuid().toHexString();
        TRACE.infoT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "The running MCF with GUID {0} will be stopped now",
                new Object[] { controlledMcfGuid.toString() });

        try {
            // Add cleanup of ressources here if necessary
            this.stopMCF();
        } catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
        }
        TRACE.infoT(SIGNATURE, XIAdapterCategories.CONNECT_AF, "MCF with GUID {0} was stopped successfully.", new Object[] { controlledMcfGuid.toString() });
        TRACE.exiting(SIGNATURE);
    }

    // CS_MCFASTOP END

    /**
     * Checks whether this mcf has a running inbound processing thread
     * 
     * @return true if running. If false is returned no inbound messages can be
     *         sent (ra implementation specific)
     **/
    public boolean isRunning() {
        if (threadStatus == TH_STARTED)
            return true;
        else
            return false;
    }

}
