/*
 * (C) 2006 SAP XI 7.1 Adapter Framework Resource Adapter Skeleton
 */

package com.mulesoft.adapter.ra;

import javax.resource.NotSupportedException;
import javax.resource.ResourceException;
import javax.resource.cci.InteractionSpec;
import javax.resource.cci.Record;
import javax.resource.cci.ResourceWarning;

import com.mulesoft.adapter.helper.IPILogger;
import com.sap.aii.af.lib.ra.cci.XIInteraction;
import com.sap.aii.af.lib.ra.cci.XIInteractionSpec;
import com.sap.aii.af.lib.ra.cci.XIMessageRecord;
import com.sap.engine.interfaces.messaging.api.Message;

/**
 * <code>CciInteraction</code> represents the XI 3.0 AF compliant interaction implementation.
 * It processes the message processing for the outbound (AF->ra) direction.
 * @version: $Id: //tc/xpi.external/NW730EXT_01_REL/src/_sample_rar_module/rar/src/com/sap/aii/af/sample/adapter/ra/CCIInteraction.java#1 $
 **/

public class CCIInteraction implements XIInteraction {

    private static final XITrace TRACE = new XITrace(CCIInteraction.class.getName());
    private javax.resource.cci.Connection connection;
    private SPIManagedConnection mc = null;
    private SPIManagedConnectionFactory mcf = null;

    /** @link dependency
     * @stereotype instantiate*/
    /*# CciConnection lnkCciInteraction; */

    /**
     * Creates a new CCI interaction object to call "methods" of an ra
     * The constructor is called by the CciConnection only.
     * (ra implementation specific)
     *
     * @param cciConnection The corresponding connection on which this interaction takes place
     * @throws ResourceException Thrown if message factory cannot be instantiated
     */
    public CCIInteraction(javax.resource.cci.Connection cciConnection) throws ResourceException {
        final String SIGNATURE = "CciInteraction(javax.resource.cci.Connection)";
        TRACE.entering(SIGNATURE, new Object[] {cciConnection});

        // Get access to the referred managed connection (factory)
        if (cciConnection == null) {
                ResourceException re = new ResourceException("No related CCI connection in Interaction (cciConnection is null).");
                TRACE.throwing(SIGNATURE, re);
                throw re;
        }

        this.connection = cciConnection;
        this.mc = ((com.mulesoft.adapter.ra.CCIConnection)connection).getManagedConnection();

        if (mc == null) {
                ResourceException re = new ResourceException("No related managed connection in CCI connection (mc is null).");
                TRACE.throwing(SIGNATURE, re);
                throw re;
        }

        this.mcf = (SPIManagedConnectionFactory) mc.getManagedConnectionFactory();

        if (mcf == null) {
                ResourceException re = new ResourceException("No related managed connection factory in managed connection (mcf is null).");
                TRACE.throwing(SIGNATURE, re);
                throw re;
        }

        TRACE.exiting(SIGNATURE);
    }

    /**
     * Returns the CCI connection that is related to this interaction
     * (CCI JCA 1.0)
     *
     * @return connection		The related connection
     */
    public javax.resource.cci.Connection getConnection() {
        return connection;
    }

    /**
     * Disconnects the current interaction from its current CCI connection
     * (CCI JCA 1.0)
     */
    public void close() throws ResourceException {
        final String SIGNATURE = "close()";
        TRACE.entering(SIGNATURE);
        connection = null;
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Execute the interaction. This flavour of the execute() signature is mapped to the second flavor
     * For details see: public Record execute (InteractionSpec ispec, Record input)
     * (CCI JCA 1.0)
     *
     * @param ispec		Defines the interaction to be executed
     * @param input		The input CCI record related to the ispec
     * @param output	The output CCI record related to the ispec
     * @return True if execution was succesful
     */
    public boolean execute (InteractionSpec ispec, Record input, Record output) throws ResourceException {

        final String SIGNATURE = "execute(InteractionSpec ispec, Record input, Record output)";
        TRACE.entering(SIGNATURE);

        if (!(output instanceof XIMessageRecord)) {
            ResourceException re = new ResourceException("Output record is no XI AF XIMessageRecord.");
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }

        XIMessageRecord callerOutput = (XIMessageRecord) output;
        XIMessageRecord localOutput = (XIMessageRecord) execute(ispec, input);

        try {
            callerOutput.setXIMessage(localOutput.getXIMessage());
            callerOutput.setRecordName(localOutput.getRecordName());
            callerOutput.setRecordShortDescription(localOutput.getRecordShortDescription());
        }
        catch (Exception e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.errorT(SIGNATURE, XIAdapterCategories.CONNECT,"SOA.apt_sample.0002",  "Exception during output record transfer. Reason: {0}", new Object[] { e.getMessage()});
            ResourceException re = new ResourceException("Output record cannot be filled. Reason: " + e.getMessage());
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }


        TRACE.exiting(SIGNATURE);
        return true;
    }

    /**
     * Execute the interaction. The execution is controlled by the InteractionSpec.
     * The standard XI AF CCI usage requires that the InteractionSpec is a XiInteractionSpec
     * This sample implementation dumps every interAction to file and gives back an appropiate output record.
     * (CCI JCA 1.0)
     *
     * @param ispec		Defines the interaction to be executed
     * @param input		The input CCI record related to the ispec
     * @return			The output CCI record related to the ispec
     * @throws ResourceException	Thrown if no connection, wrong input
     */
    public Record execute(InteractionSpec ispec, Record input) throws ResourceException
    {
        final String SIGNATURE = "execute(InteractionSpec ispec, Record input)";
        TRACE.entering(SIGNATURE, new Object[]{ispec, input});

        // Check InteractionSpec
        if (ispec == null) {
            ResourceException re = new ResourceException("Input ispec is null.");
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }

        if (!(ispec instanceof XIInteractionSpec))
        {
            ResourceException re = new ResourceException("Input ispec is no XI AF InteractionSpec.");
            TRACE.throwing(SIGNATURE, re);
            throw re;
        }

        try {
            XIMessageRecordImpl messageRecord = (XIMessageRecordImpl) input;
            Message xiMessage = messageRecord.getXIMessage();
            IPILogger logger = mc.createLogger(xiMessage);
            return this.mc.getModule().execute(ispec, messageRecord, logger);
        } catch (ResourceException e) {
            TRACE.warningT(SIGNATURE, XIAdapterCategories.CONNECT_EIS, "Exception during PIModule execution. Reason: {0}", new Object[] { e.getMessage()});
            TRACE.throwing(SIGNATURE, e);
            throw e;
        }
    }

    /**
     * Returns list of warnings occured in this interaction. This sample does not support warnings
     * hence it returns null
     * (CCI JCA 1.0)
     *
     * @return			ResourceWarnings occured in this interaction
     * @throws ResourceException	Never thrown
     */
    public ResourceWarning getWarnings() throws ResourceException {
       return null;
    }

    /**
     * Resets warning list in this interaction. It is not supported in this sample.
     * (CCI JCA 1.0)
     */
    public void clearWarnings() throws ResourceException {
        // nop
    }

    /**
     * Returns the NetWeaver XI specific InteractionSpec
     * (XI specific)
     * @see com.sap.aii.af.lib.ra.cci.XIInteraction#getXIInteractionSpec(java.lang.String)
     *
     * @return The XI specific InteractionSpec
     */
    public XIInteractionSpec getXIInteractionSpec() throws NotSupportedException {
            return new XIInteractionSpecImpl();
    }

}