/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.resource.ResourceException;

import org.mule.api.MuleException;
import org.mule.tools.module.invocation.DynamicModule;
import org.w3c.dom.Document;

import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.XML;
import com.mulesoft.adapter.module.AbstractOperationHandler;
import com.mulesoft.adapter.module.PIMessageParameters;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.engine.interfaces.messaging.api.auditlog.AuditLogStatus;
import com.sforce.async.BatchInfo;
import com.sforce.async.BatchResult;
import com.sforce.soap.partner.UpsertResult;

/**
 * Provide support for `upsert` operation.
 * 
 * Expected input format:
 * 
 * <externalIDFieldName>SomeId__c</externalIDFieldName> <sObjects>
 * <type>Account</type> <Name></Name> </sObjects>
 * 
 * or
 * 
 * <externalIDFieldName>SomeId__c</externalIDFieldName> <sObjects
 * xsi:type="ns3:Account" xmlns:ns="urn:sobject.enterprise.soap.sforce.com">
 * <ns:Name></ns:Name> </sObjects>
 * 
 * @see http
 *      ://www.salesforce.com/us/developer/docs/api/Content/sforce_api_calls_upsert
 *      .htm
 */
public class UpsertOperationHandler extends AbstractOperationHandler {

    private static final String PROCESSOR_NAME = "upsert";
    private static final String BULK_PROCESSOR_NAME = "upsert-bulk";
    private static final String EXTERNAL_ID_FIELD_NAME_PARAMETER_NAME = "externalIdFieldName";
    private static final String EXTERNAL_ID_FIELD_NAME_TAG_NAME = "externalIDFieldName";
    private static final String TYPE_PARAMETER_NAME = "type";
    private static final String OBJECTS_PARAMETER_NAME = "objects";
    private static final int MAX_RECORDS_NUMBER = 200;

    public UpsertOperationHandler(final DynamicModule module) {
        super(UpsertOperationHandler.PROCESSOR_NAME, module);
    }

    protected final void upsert(final String externalIdFieldName, final String type, final List<Map<String, Object>> objects, IPILogger logger)
            throws MuleException, ResourceException {
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(UpsertOperationHandler.EXTERNAL_ID_FIELD_NAME_PARAMETER_NAME, externalIdFieldName);
        parameters.put(UpsertOperationHandler.TYPE_PARAMETER_NAME, type);
        parameters.put(UpsertOperationHandler.OBJECTS_PARAMETER_NAME, objects);

        logger.reportProcessingStatus(ProcessState.OK, "invoking operation");
        final List<UpsertResult> results = invoke(parameters);
        logger.reportProcessingStatus(ProcessState.OK, "invocation finished");
        Results.throwIfUpsertNotSuccessful(results);
    }

    protected final void upsertBulk(final String externalIdFieldName, final String type, final List<Map<String, String>> objects, IPILogger logger)
            throws MuleException, ResourceException, InterruptedException {
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(UpsertOperationHandler.EXTERNAL_ID_FIELD_NAME_PARAMETER_NAME, externalIdFieldName);
        parameters.put(UpsertOperationHandler.TYPE_PARAMETER_NAME, type);
        parameters.put(UpsertOperationHandler.OBJECTS_PARAMETER_NAME, objects);

        logger.reportProcessingStatus(ProcessState.OK, "invoking bulk operation");
        final BatchInfo batchInfo = invoke(UpsertOperationHandler.BULK_PROCESSOR_NAME, parameters);
        logger.reportProcessingStatus(ProcessState.OK, "awaiting bulk operation completion");
        final BatchResult batchResult = Bulks.awaitCompletion(getModule(), batchInfo);
        logger.reportProcessingStatus(ProcessState.OK, "invocation finished");
        Bulks.throwIfNotSuccessful(batchResult);
    }

    @Override
    public final byte[] handle(final PIMessageParameters messageParameters) throws MuleException, ResourceException {
        Document payload = messageParameters.getRequestContent();
        Map<String, List<Map<String, String>>> typeObjectMap = SObjects.parse(payload);

        if (typeObjectMap.isEmpty()) {
            throw new IllegalArgumentException("No SObjects identified.");
        }
        if (typeObjectMap.size() > 1) {
            throw new IllegalArgumentException("More than a single type has been sent.");
        }
        final String externalIdFieldName = XML.firstNamedTextNode(payload, UpsertOperationHandler.EXTERNAL_ID_FIELD_NAME_TAG_NAME);
        if (externalIdFieldName == null) {
            throw new IllegalArgumentException("null externalIdFieldName");
        }

        final Map.Entry<String, List<Map<String, String>>> entry = typeObjectMap.entrySet().iterator().next();
        final String type = entry.getKey();
        final List<Map<String, String>> objects = entry.getValue();
        int size = objects.size();

        IPILogger logger = messageParameters.getLogger();
        logger.reportAuditStatus(AuditLogStatus.SUCCESS, "Message of type {0} contains {1} sObjects", type, size);

        if (size <= UpsertOperationHandler.MAX_RECORDS_NUMBER) {
            // create uses a different representation of e.g. date
            final List<Map<String, Object>> typedTypeObjectsMap = SObjects.asTypedObject(getModule(), type, objects);
            upsert(externalIdFieldName, type, typedTypeObjectsMap, logger);
        } else {
            try {
                upsertBulk(externalIdFieldName, type, objects, logger);
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                return null;
            }
        }
        return null;
    }

}