/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.mule.api.MuleException;
import org.mule.tools.module.invocation.DynamicModule;
import org.mule.util.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.mulesoft.adapter.helper.XML;
import com.sforce.ws.bind.TypeInfo;
import com.sforce.ws.bind.TypeMapper;
import com.sforce.ws.parser.XmlOutputStream;
import com.sforce.ws.wsdl.Constants;

public final class SObjects {

    private static final String SALESFORCE_ENTERPRISE_NAMESPACE = "urn:enterprise.soap.sforce.com";
    private static final String OBJECTS_NAMESPACE = "sObjects";
    static final String TYPE_ATTRIBUTE_NAME = "type";
    private static final TypeMapper TYPE_MAPPER = new TypeMapper();

    private SObjects() {
    }

    /**
     * @param node
     * @param attributes
     * @return type of specified SObject (represented by specified {@link Node}
     *         and `attributes`). Support both enterprise and partner
     *         representations.
     */
    public static String extractType(final Node node, final Map<String, String> attributes) {
        if (node.hasAttributes()) {
            final NamedNodeMap nodeAttributes = node.getAttributes();
            for (int i = 0; i < nodeAttributes.getLength(); i++) {
                final Node nodeAttribute = nodeAttributes.item(i);
                if (SObjects.TYPE_ATTRIBUTE_NAME.equals(XML.stripNamespaceIfNecessary(nodeAttribute.getNodeName()))) {
                    return XML.stripNamespaceIfNecessary(nodeAttribute.getNodeValue());
                }
            }
        }
        return attributes.remove(SObjects.TYPE_ATTRIBUTE_NAME);
    }

    /**
     * @param payload
     * @return
     */
    public static Map<String, List<Map<String, String>>> parse(final Document payload) {
        final Map<String, List<Map<String, String>>> typedObjects = new HashMap<String, List<Map<String, String>>>();
        List<Node> sObjectNodes = SObjects.findSObjectRootNodes(payload);

        for (final Node rootNode : sObjectNodes) {
            final Map<String, String> attributes = extractSObjectAttributes(rootNode);

            final String type = SObjects.extractType(rootNode, attributes);
            if (type == null) {
                throw new IllegalArgumentException("Cannot access type attribute");
            }

            addObjects(type, typedObjects, attributes);

        }
        return typedObjects;
    }

    private static void addObjects(final String type, final Map<String, List<Map<String, String>>> typedObjects, final Map<String, String> attributes) {
        List<Map<String, String>> objectsOfType = typedObjects.get(type);
        if (objectsOfType == null) {
            objectsOfType = new ArrayList<Map<String, String>>();
            typedObjects.put(type, objectsOfType);
        }
        objectsOfType.add(attributes);
    }

    /**
     * convert element name/value in XML to name/value in map
     */
    private static Map<String, String> extractSObjectAttributes(final Node rootNode) {
        NodeList attributeNodes = rootNode.getChildNodes();
        int attributeNodesLength = attributeNodes.getLength();
        final Map<String, String> attributes = new HashMap<String, String>(attributeNodesLength);

        for (int i = 0; i < attributeNodesLength; i++) {
            final Node node = attributeNodes.item(i);
            final String key = XML.stripNamespaceIfNecessary(node.getNodeName());

            attributes.put(key, node.getTextContent());
        }

        return attributes;
    }

    /**
     * Appends XML representation of specified `object` to specified
     * {@link XmlOutputStream}.
     * 
     * @param outputStream
     * @param object
     * @throws IOException
     */
    private static void generate(String tagname, final XmlOutputStream outputStream, final Map<String, Object> object) throws IOException {
        outputStream.writeStartTag("", tagname);
        final String type = (String) object.remove(SObjects.TYPE_ATTRIBUTE_NAME);
        if (type != null) {
            outputStream.writeAttribute("", "xsi:type", "sf:" + type);
        }
        outputStream.writeAttribute("", "xmlns:sf", "urn:sobject.enterprise.soap.sforce.com");
        for (final Map.Entry<String, Object> entry : object.entrySet()) {
            final String tagName = entry.getKey();
            final TypeInfo info = new TypeInfo("", "sf:" + tagName, "", "", 1, 1, true);
            SObjects.TYPE_MAPPER.writeObject(outputStream, info, entry.getValue(), true);//
        }
        outputStream.writeEndTag("", tagname);
    }

    /**
     * <queryResponse xmlns="urn:enterprise.soap.sforce.com"> <result>
     * <done>true</done> <queryLocator>01g30000000590JAAQ-3</queryLocator>
     * <records xsi:type="sf:Contact"
     * xmlns:sf="urn:sobject.enterprise.soap.sforce.com">
     * <sf:FirstName>Merce</sf:FirstName> <sf:LastName>Carroll</sf:LastName>
     * </records> <size>1</size> </result> </queryResponse>
     */
    public static byte[] generateQueryResponse(final List<Map<String, Object>> objects, String rootElementName, String rootElementNamespace) {
        try {
            final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            final XmlOutputStream xmlOutputStream = new XmlOutputStream(outputStream, false);

            xmlOutputStream.startDocument();
            xmlOutputStream.writeStartTag(rootElementNamespace, rootElementName);
            if (!StringUtils.isEmpty(rootElementNamespace)) {
                xmlOutputStream.writeAttribute("", "xmlns", SALESFORCE_ENTERPRISE_NAMESPACE);
            }

            xmlOutputStream.writeAttribute("", "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");

            xmlOutputStream.writeStartTag("", "result");
            xmlOutputStream.writeStartTag("", "done");
            xmlOutputStream.writeText("true");
            xmlOutputStream.writeEndTag("", "done");

            xmlOutputStream.writeStartTag("", "queryLocator");
            xmlOutputStream.writeEndTag("", "queryLocator");

            for (final Map<String, Object> object : objects) {
                SObjects.generate("records", xmlOutputStream, object);
            }

            xmlOutputStream.writeStartTag("", "size");
            xmlOutputStream.writeText(Integer.toString(objects.size()));
            xmlOutputStream.writeEndTag("", "size");

            xmlOutputStream.writeEndTag("", "result");

            xmlOutputStream.writeEndTag(rootElementNamespace, rootElementName);

            xmlOutputStream.endDocument();
            xmlOutputStream.close();
            return outputStream.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("IO-Error when constructing XML from sObject-list");
        }
    }

    /**
     * <?xml version="1.0" encoding="UTF-8"?> <ns0:subscribeContactResponse
     * xmlns:ns0="urn:enterprise.soap.sforce.com"> <ns0:createdDate/>
     * <ns0:channel/> <ns0:type>Contact</ns0:type> <ns0:sObjects>
     * <ns1:fieldsToNull xmlns:ns1="urn:sobject.enterprise.soap.sforce.com"/>
     * <ns1:Id xmlns:ns1="urn:sobject.enterprise.soap.sforce.com"/> <ns1:type
     * xmlns:ns1="urn:sobject.enterprise.soap.sforce.com"/> <ns1:Account
     * xmlns:ns1="urn:sobject.enterprise.soap.sforce.com"> </ns0:sObjects>
     * </ns0:subscribeContactResponse>
     **/
    public static byte[] generateSubscriptionResponse(final Map<String, Object> object, String rootElementName, String rootElementNamespace, String channel,
            String createdDate, String type) {
        try {
            final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            final XmlOutputStream xmlOutputStream = new XmlOutputStream(outputStream, false);

            xmlOutputStream.startDocument();
            xmlOutputStream.writeStartTag(rootElementNamespace, rootElementName);
            if (!StringUtils.isEmpty(rootElementNamespace)) {
                xmlOutputStream.writeAttribute("", "xmlns", SALESFORCE_ENTERPRISE_NAMESPACE);
            }

            xmlOutputStream.writeAttribute("", "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");

            xmlOutputStream.writeStringElement(SALESFORCE_ENTERPRISE_NAMESPACE, "channel", channel);
            xmlOutputStream.writeStringElement(SALESFORCE_ENTERPRISE_NAMESPACE, "createdDate", createdDate);
            xmlOutputStream.writeStringElement(SALESFORCE_ENTERPRISE_NAMESPACE, "type", type);

            SObjects.generate("sObjects", xmlOutputStream, object);

            xmlOutputStream.writeEndTag(rootElementNamespace, rootElementName);

            xmlOutputStream.endDocument();
            xmlOutputStream.close();
            return outputStream.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("IO-Error when constructing XML from sObject-list");
        }
    }

    /**
     * @param document
     * @return all root {@link Node}s
     */
    public static List<Node> findSObjectRootNodes(final Document document) {
        final List<Node> nodes = new ArrayList<Node>();

        Node firstChild = document.getFirstChild();
        final NodeList children = firstChild.getChildNodes();
        int childNodesLength = children.getLength();

        for (int i = 0; i < childNodesLength; i++) {
            final Node node = children.item(i);
            if (node.getNodeName().endsWith(SObjects.OBJECTS_NAMESPACE)) {
                nodes.add(node);
            }
        }
        return nodes;
    }

    /**
     * Specified `objects` as typed objects.
     * 
     */
    public static List<Map<String, Object>> asTypedObject(final DynamicModule module, final String type, final List<Map<String, String>> objects)
            throws MuleException {
        final List<Map<String, Object>> typedObjects = new ArrayList<Map<String, Object>>(objects.size());

        final SObjectTypeFinder typeFinder = new SObjectTypeFinder(type, module);
        typeFinder.init();

        for (final Map<String, String> object : objects) {
            final Map<String, Object> typedObject = new HashMap<String, Object>();

            for (final Map.Entry<String, String> attributeEntry : object.entrySet()) {
                final String attributeName = attributeEntry.getKey();
                // TODO: check if this has to be invoked for all of the objects
                // of the same type
                final String attributeType = typeFinder.findField(attributeName).getType().name();
                final Object value = SObjects.TYPE_MAPPER.deserialize(attributeEntry.getValue(), new QName(Constants.SCHEMA_NS, attributeType));
                typedObject.put(attributeName, value);
            }
            typedObjects.add(typedObject);
        }
        
        return typedObjects;
    }

}