/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.loader;

import static org.mule.extension.maven.ExtensionDescriptorMojo.XML_BASED_EXTENSION_MODEL_LOADER;
import static org.mule.maven.pom.parser.api.model.MavenModelBuilderProvider.discoverProvider;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.container.api.ContainerClassLoaderProvider.createContainerClassLoader;
import static org.mule.runtime.container.api.ModuleRepository.createModuleRepository;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.util.UUID.getUUID;
import static org.mule.runtime.core.internal.util.jar.JarLoadingUtils.loadFileContentFrom;
import static org.mule.runtime.globalconfig.api.GlobalConfigLoader.setMavenConfig;
import static org.mule.runtime.module.artifact.activation.api.classloader.ArtifactClassLoaderResolver.classLoaderResolver;
import static org.mule.runtime.module.artifact.activation.api.descriptor.DeployableArtifactDescriptorFactory.defaultArtifactDescriptorFactory;
import static org.mule.runtime.module.artifact.activation.api.extension.discovery.ExtensionModelDiscoverer.defaultExtensionModelDiscoverer;
import static org.mule.runtime.module.artifact.activation.api.extension.discovery.ExtensionModelLoaderRepository.getExtensionModelLoaderManager;
import static org.mule.runtime.module.artifact.activation.api.plugin.PluginDescriptorResolver.pluginDescriptorResolver;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor.MULE_ARTIFACT_JSON_DESCRIPTOR;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.MULE_LOADER_ID;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor.MULE_ARTIFACT_PATH_INSIDE_JAR;
import static org.mule.runtime.module.deployment.impl.internal.maven.AbstractMavenClassLoaderConfigurationLoader.CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static com.google.common.io.Files.createTempDir;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.maven.artifact.ArtifactUtils.key;

import org.mule.extension.maven.ExtensionDescriptorMojo;
import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenModelBuilder;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.internal.model.MavenPomModelWrapper;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptorBuilder;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.extension.api.loader.ExtensionModelLoader;
import org.mule.runtime.module.artifact.activation.api.classloader.ArtifactClassLoaderResolver;
import org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModel;
import org.mule.runtime.module.artifact.activation.api.extension.discovery.ExtensionDiscoveryRequest;
import org.mule.runtime.module.artifact.activation.api.extension.discovery.ExtensionModelLoaderRepository;
import org.mule.runtime.module.artifact.activation.api.plugin.PluginModelResolver;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorCreateException;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.tools.api.classloader.model.ArtifactCoordinates;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;

import org.apache.commons.io.FileUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.execution.MavenExecutionRequest;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Model;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Loader of an {@link ExtensionModel} for a Mule plugin artifact from an extension {@link MavenProject}.
 *
 * @since 1.0
 */
public class MavenProjectExtensionModelLoader {

  private final Log log;

  public MavenProjectExtensionModelLoader(Log log) {
    this.log = log;
  }

  /**
   * Builds an {@link ExtensionModel} from a {@link MavenProject} and a {@link MulePluginModel} that holds the information about
   * the extension mule plugin that is wanted to be loaded.
   *
   * @return an {@link ExtensionModel} built up from the extension {@link MavenProject} provided.
   * @throws MojoFailureException
   */
  public ExtensionModel loadExtension(MavenProject project, MavenSession session)
      throws MojoFailureException, MojoExecutionException {
    String artifactId = project.getArtifactId();
    log.debug(format("Creating ExtensionModel for:[%s]", artifactId));
    configureMavenClient(session);

    Predicate<PluginDescriptor> enableXmlValidations =
        (pluginDescriptor) -> pluginDescriptor.getPluginFile().equals(project.getArtifact().getFile()) &&
            XML_BASED_EXTENSION_MODEL_LOADER.equals(pluginDescriptor.getMuleArtifactLoaderDescriptor().getId());

    ArtifactClassLoader muleContainerClassloader = createMuleContainerClassloader();

    PluginFileMavenReactor mavenReactor = new PluginFileMavenReactor(project);
    Map<String, Object> classLoaderModelAttributes = new HashMap<>();
    classLoaderModelAttributes.put(CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER, mavenReactor);

    // TODO review this project.getArtifact() vs project.getGroupId()... etc (which uses the parent pom if not a value at the pom
    // model level)
    return withPluginClassLoaders(project.getArtifact(), classLoaderModelAttributes,
                                  muleContainerClassloader,
                                  of(mavenReactor),
                                  of(project.getModel()),
                                  enableXmlValidations,
                                  artifactClassLoader -> doLoadExtensionModel(project.getArtifact(),
                                                                              artifactClassLoader));
  }

  public ExtensionModel loadExtension(Artifact artifact, MavenSession session)
      throws MojoFailureException, MojoExecutionException {
    log.debug(format("Creating ExtensionModel for:[%s]", artifact.getArtifactId()));
    configureMavenClient(session);

    ArtifactClassLoader muleContainerClassloader = createMuleContainerClassloader();

    Map<String, Object> classLoaderModelAttributes = new HashMap<>();

    return withPluginClassLoaders(artifact, classLoaderModelAttributes,
                                  muleContainerClassloader,
                                  empty(),
                                  empty(),
                                  pluginDescriptor -> false,
                                  artifactClassLoader -> doLoadExtensionModel(artifact,
                                                                              artifactClassLoader));
  }

  private Set<ExtensionModel> discoverPluginsExtensionModel(MuleDeployableArtifactClassLoader artifactClassLoader,
                                                            ExtensionModelLoaderRepository extensionModelLoaderRepository) {
    List<ArtifactPluginDescriptor> artifacts = artifactClassLoader.getArtifactPluginClassLoaders()
        .stream()
        .map(a -> (ArtifactPluginDescriptor) a.getArtifactDescriptor())
        .collect(toList());
    return defaultExtensionModelDiscoverer(artifactClassLoader, extensionModelLoaderRepository)
        .discoverPluginsExtensionModels(ExtensionDiscoveryRequest.builder()
            .setArtifactPlugins(artifacts).build());
  }


  private void configureMavenClient(MavenSession session) {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = MavenConfiguration.newMavenConfigurationBuilder();
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());
    MavenExecutionRequest request = session.getRequest();

    mavenConfigurationBuilder.localMavenRepositoryLocation(request.getLocalRepositoryPath());

    mavenConfigurationBuilder.userSettingsLocation(request.getUserSettingsFile());
    mavenConfigurationBuilder.globalSettingsLocation(request.getGlobalSettingsFile());
    settingsSupplierFactory.environmentSettingsSecuritySupplier().ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);

    mavenConfigurationBuilder.userProperties(request.getUserProperties());
    mavenConfigurationBuilder.activeProfiles(request.getActiveProfiles());
    mavenConfigurationBuilder.inactiveProfiles(request.getInactiveProfiles());

    mavenConfigurationBuilder.ignoreArtifactDescriptorRepositories(false);

    setMavenConfig(mavenConfigurationBuilder.build());
  }

  class PluginFileMavenReactor implements MavenReactorResolver {

    private final MavenProject project;

    public PluginFileMavenReactor(MavenProject project) {
      this.project = project;
    }

    @Override
    public File findArtifact(BundleDescriptor bundleDescriptor) {
      if (checkArtifact(bundleDescriptor)) {
        if (bundleDescriptor.getType().equals("pom")) {
          return project.getFile();
        } else {
          return project.getArtifact().getFile();
        }
      }
      return null;
    }

    private boolean checkArtifact(BundleDescriptor bundleDescriptor) {
      return bundleDescriptor.getGroupId().equals(project.getArtifact().getGroupId())
          && bundleDescriptor.getArtifactId().equals(project.getArtifact().getArtifactId())
          && bundleDescriptor.getVersion().equals(project.getArtifact().getVersion());
    }

    @Override
    public List<String> findVersions(BundleDescriptor bundleDescriptor) {
      if (checkArtifact(bundleDescriptor)) {
        return singletonList(project.getArtifact().getVersion());
      }
      return emptyList();
    }
  }

  private ExtensionModel doLoadExtensionModel(Artifact artifact, MuleDeployableArtifactClassLoader artifactClassLoader)
      throws MojoFailureException {
    ExtensionModelLoaderRepository extensionModelLoaderRepository = getExtensionModelLoaderManager();
    try {
      startIfNeeded(extensionModelLoaderRepository);
    } catch (MuleException e) {
      throw new MojoFailureException(format("Failure while looking implementation classes of [%s] class through SPI",
                                            ExtensionModelLoader.class.getName()),
                                     e);
    }
    Set<ExtensionModel> discoverPluginsExtensionModel =
        discoverPluginsExtensionModel(artifactClassLoader, extensionModelLoaderRepository);
    Optional<ExtensionModel> extensionModelOptional = discoverPluginsExtensionModel.stream()
        .filter(extensionModel -> extensionModel.getArtifactCoordinates()
            .map(artifactCoordinates -> artifactCoordinates.getGroupId().equals(artifact.getGroupId()) &&
                artifactCoordinates.getArtifactId().equals(artifact.getArtifactId()) &&
                artifactCoordinates.getVersion().equals(artifact.getVersion())

            ).orElse(false))
        .findFirst();

    if (!extensionModelOptional.isPresent()) {
      List<String> foundExtensionModels =
          discoverPluginsExtensionModel.stream().map(ExtensionModel::getName).collect(toList());
      if (foundExtensionModels.isEmpty()) {
        throw new MojoFailureException(format("Could not load Extension Model for %s", key(artifact)));
      }
      throw new MojoFailureException(format("Could not load Extension Model for %s, the following were loaded: %n%s",
                                            key(artifact), join(format(", %n"), foundExtensionModels)));
    }

    return extensionModelOptional.get();
  }

  private <T> T withPluginClassLoaders(Artifact artifact, Map<String, Object> classLoaderModelLoaderAttributes,
                                       ArtifactClassLoader containerArtifactClassLoader,
                                       Optional<MavenReactorResolver> mavenReactorResolver,
                                       Optional<Model> pomModel,
                                       Predicate<PluginDescriptor> enableXmlValidations,
                                       Action<T> action)
      throws MojoFailureException {
    String uuid = getUUID();
    String applicationName = uuid + "-extension-model-temp-app";

    File tempFolder = null;
    try {
      tempFolder = createTempDirectory(null).toFile();
      File applicationFolder = new File(tempFolder, applicationName);

      createPomFile(artifact, uuid, applicationFolder);

      MuleApplicationModel muleApplicationModel = new MuleApplicationModel.MuleApplicationModelBuilder()
          .setMinMuleVersion(getProductVersion())
          .setName(applicationName)
          .setRequiredProduct(MULE)
          .withBundleDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()))
          .withClassLoaderModelDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID,
                                                                                 classLoaderModelLoaderAttributes))
          .build();

      Map<ArtifactCoordinates, Supplier<MavenPomModel>> pomModelMap =
          pomModel
              .map(pom -> singletonMap(new ArtifactCoordinates(artifact.getGroupId(), artifact.getArtifactId(),
                                                               artifact.getVersion()),
                                       (Supplier<MavenPomModel>) () -> new MavenPomModelWrapper(pom)))
              .orElse(emptyMap());

      DeployableProjectModel model =
          new MavenExtensionDeployableProjectModelBuilder(applicationFolder, of(muleApplicationModel),
                                                          mavenReactorResolver, pomModelMap).build();

      ApplicationDescriptor artifactDescriptor =
          defaultArtifactDescriptorFactory().createApplicationDescriptor(model, emptyMap(),
                                                                         getPluginModelResolver(enableXmlValidations),
                                                                         pluginDescriptorResolver());

      File moduleDiscovererTemporaryFolder = new File(tempFolder, ".moduleDiscoverer");
      if (!moduleDiscovererTemporaryFolder.mkdir()) {
        throw new MojoFailureException("Error while generating class loaders in order to load the Extension Model, cannot create directory "
            + moduleDiscovererTemporaryFolder.getAbsolutePath());
      }

      ArtifactClassLoaderResolver artifactClassLoaderResolver =
          classLoaderResolver(containerArtifactClassLoader,
                              createModuleRepository(ArtifactClassLoaderResolver.class.getClassLoader(),
                                                     moduleDiscovererTemporaryFolder),
                              (empty) -> applicationFolder);

      MuleDeployableArtifactClassLoader artifactClassLoader =
          artifactClassLoaderResolver.createApplicationClassLoader(artifactDescriptor);

      try {
        return action.call(artifactClassLoader);
      } catch (Exception e) {
        throw new MojoFailureException("Error while generating class loaders in order to load the Extension Model", e);
      } finally {
        if (artifactClassLoader != null) {
          artifactClassLoader.dispose();
        }
      }
    } catch (IOException e) {
      throw new MojoFailureException("Error while generating class loaders in order to load the Extension Model", e);
    } finally {
      deleteQuietly(tempFolder);
    }
  }


  private PluginModelResolver getPluginModelResolver(Predicate<PluginDescriptor> enableXmlValidations) {

    return (bundleDependency) -> {
      try {
        File pluginFile = new File(bundleDependency.getBundleUri());
        Optional<byte[]> jsonDescriptorContentOptional = empty();
        String mulePluginJsonPath = MULE_ARTIFACT_PATH_INSIDE_JAR + "/" + MULE_ARTIFACT_JSON_DESCRIPTOR;
        jsonDescriptorContentOptional =
            getMuleArtifactJsonDescriptorContent(pluginFile, jsonDescriptorContentOptional, mulePluginJsonPath);

        MulePluginModel artifactModel = jsonDescriptorContentOptional
            .map(jsonDescriptorContent -> new MulePluginModelJsonSerializer()
                .deserialize(new String(jsonDescriptorContent, UTF_8)))
            .orElseThrow(
                         () -> new ArtifactDescriptorCreateException(format("The plugin descriptor '%s' on plugin file '%s' is not present",
                                                                            mulePluginJsonPath, pluginFile)));

        // Create another plugin model to set as required product MULE and minMuleVersion 4.1.1 due to validations cannot be
        // disabled
        MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder();
        mulePluginModelBuilder
            .setName(artifactModel.getName())
            .setMinMuleVersion("4.1.1")
            .setRequiredProduct(Product.MULE)
            .withBundleDescriptorLoader(artifactModel.getBundleDescriptorLoader())
            .withClassLoaderModelDescriptorLoader(artifactModel.getClassLoaderModelLoaderDescriptor());

        artifactModel.getLicense().ifPresent(licenseModel -> mulePluginModelBuilder.withLicenseModel()
            .setAllowsEvaluationLicense(licenseModel.isAllowsEvaluation())
            .setProvider(licenseModel.getProvider())
            .setRequiredEntitlement(licenseModel.getRequiredEntitlement().orElse(null)));

        artifactModel.getExtensionModelLoaderDescriptor().ifPresent(extensionModelDescriptorLoader -> {
          MuleArtifactLoaderDescriptorBuilder muleArtifactLoaderDescriptorBuilder =
              mulePluginModelBuilder.withExtensionModelDescriber();
          muleArtifactLoaderDescriptorBuilder.setId(extensionModelDescriptorLoader.getId());
          extensionModelDescriptorLoader.getAttributes().entrySet().stream()
              .forEach(entry -> muleArtifactLoaderDescriptorBuilder.addProperty(entry.getKey(), entry.getValue()));

          // If the current plugin being built is an XML SDK we should always validate its XML
          if (artifactModel.getExtensionModelLoaderDescriptor().isPresent() &&
              enableXmlValidations
                  .test(new PluginDescriptor(pluginFile, artifactModel.getExtensionModelLoaderDescriptor().get()))) {
            muleArtifactLoaderDescriptorBuilder.addProperty(ExtensionDescriptorMojo.VALIDATE_XML, true);
          }
        });

        return mulePluginModelBuilder.build();
      } catch (IOException e) {
        throw new ArtifactDescriptorCreateException(e);
      }
    };
  }


  private Optional<byte[]> getMuleArtifactJsonDescriptorContent(File pluginFile,
                                                                Optional<byte[]> jsonDescriptorContentOptional,
                                                                String mulePluginJsonPath)
      throws IOException {
    if (pluginFile.isFile()) {
      checkArgument(pluginFile.getName().endsWith(".jar"),
                    "provided file is not a plugin: " + pluginFile.getAbsolutePath());
      jsonDescriptorContentOptional = loadFileContentFrom(pluginFile, mulePluginJsonPath);
    } else {
      File mulePluginJsonFile =
          pluginFile.toPath().resolve(Paths.get(MULE_ARTIFACT_PATH_INSIDE_JAR, MULE_ARTIFACT_JSON_DESCRIPTOR)).toFile();
      if (mulePluginJsonFile.exists()) {
        jsonDescriptorContentOptional = of(FileUtils.readFileToByteArray(mulePluginJsonFile));
      }
    }
    return jsonDescriptorContentOptional;
  }

  private void createPomFile(Artifact artifact, String uuid, File applicationFolder) {
    MavenModelBuilder modelBuilder =
        discoverProvider().createMavenModelBuilder(uuid, uuid, getProductVersion(), of("4.0.0"), of("mule-application"));

    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(artifact.getGroupId())
        .setArtifactId(artifact.getArtifactId())
        .setVersion(artifact.getVersion())
        .setClassifier(MULE_PLUGIN_CLASSIFIER)
        .setType("jar")
        .build();
    BundleDependency dependency = new BundleDependency.Builder().setBundleDescriptor(descriptor).build();
    modelBuilder.addDependency(dependency);

    modelBuilder.createDeployablePomFile(applicationFolder.toPath());
    MavenPomModel model = modelBuilder.getModel();

    Properties pomProperties = new Properties();
    pomProperties.setProperty("groupId", model.getGroupId());
    pomProperties.setProperty("artifactId", model.getArtifactId());
    pomProperties.setProperty("version", model.getVersion());
    modelBuilder.createDeployablePomProperties(applicationFolder.toPath(), pomProperties);
  }

  private ArtifactClassLoader createMuleContainerClassloader() {
    ModuleRepository moduleRepository = createModuleRepository(this.getClass().getClassLoader(), createTempDir());
    return createContainerClassLoader(moduleRepository, this.getClass().getClassLoader());
  }

  @FunctionalInterface
  interface Action<T> {

    T call(MuleDeployableArtifactClassLoader artifactClassLoader) throws MojoFailureException, MojoExecutionException;
  }
}
