/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.stream.Collectors.joining;
import static org.apache.maven.artifact.Artifact.SCOPE_TEST;
import static org.mule.extension.maven.ExtensionPackageMojo.MULE_PLUGIN_JSON_JAR_DESTINATION;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import static org.mule.runtime.api.dsl.DslResolvingContext.getDefault;
import org.mule.extension.maven.ExtensionPackageMojo;
import org.mule.plugin.maven.AbstractPackagePluginMojo;
import org.mule.runtime.api.deployment.meta.AbstractMuleArtifactModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.loader.ExtensionModelLoader;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Loader of an {@link ExtensionModel} for a Mule plugin artifact from an extension {@link MavenProject}.
 *
 * @since 1.0
 */
public class MavenProjectExtensionModelLoader {

  private final Log log;

  public MavenProjectExtensionModelLoader(Log log) {
    this.log = log;
  }

  /**
   * Builds an {@link ExtensionModel} from a {@link MavenProject} and a {@link MulePluginModel} that holds the information about
   * the extension mule plugin that is wanted to be loaded.
   *
   * @return an {@link ExtensionModel} built up from the extension {@link MavenProject} provided.
   * @throws MojoFailureException
   */
  public ExtensionModel loadExtension(MavenProject project, MulePluginModel mulePluginDescriber)
      throws MojoFailureException, MojoExecutionException {
    final ClassLoader pluginClassLoader = getInvokerClassLoader(project);
    final Set<ExtensionModel> extensions = getPluginsExtensions(project, pluginClassLoader);
    return getExtensionModel(pluginClassLoader, extensions, mulePluginDescriber);
  }

  /**
   * Goes over all the plugin dependencies generating the needed {@link ExtensionModel} for the current project to work properly.
   *
   * @param project to look for dependencies
   * @param pluginClassLoader class loader with all the needed JARs to load the {@link ExtensionModel}s
   * @return a set of {@link ExtensionModel} of the current plugin's dependencies
   * @throws MojoFailureException if there are plugins that could not generate an {@link ExtensionModel} despite they have a {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   * @throws MojoExecutionException if there are plugins that could not generate an {@link ExtensionModel} despite they have a {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   */
  private Set<ExtensionModel> getPluginsExtensions(MavenProject project, ClassLoader pluginClassLoader)
      throws MojoFailureException, MojoExecutionException {
    final Set<ExtensionModel> extensions = new HashSet<>();
    final Set<MulePluginModel> pluginDescriptors = filterPluginsWithExtensionModelLoader(project);

    Set<MulePluginModel> workedPlugins;
    do {
      workedPlugins = new HashSet<>();
      for (MulePluginModel mulePluginDescriber : pluginDescriptors) {
        try {
          extensions.add(getExtensionModel(pluginClassLoader, extensions, mulePluginDescriber));
          workedPlugins.add(mulePluginDescriber);
        } catch (RuntimeException e) {
          // Do nothing, as it might be a plugin that must be generated later on, we will not break.
          // Common use case is an XML plugin that depends on a plugin that hasn't generated its ExtensionModel, thus will cause a SaxParseException
          log.info(format(
                          "Couldn't resolve ExtensionModel for plugin [%s], trying again later ([%d] more plugins in the bag to work with).",
                          mulePluginDescriber.getName(), pluginDescriptors.size() - 1));
        }
      }
      pluginDescriptors.removeAll(workedPlugins);
    } while (!pluginDescriptors.isEmpty() && !workedPlugins.isEmpty());

    if (!pluginDescriptors.isEmpty()) {
      final String unresolvedPlugins = pluginDescriptors.stream()
          .map(AbstractMuleArtifactModel::getName)
          .collect(joining(","));
      throw new MojoFailureException(
                                     format("There are several plugins that are still pending to generate their ExtensionModel. Unresolved plugins: [%s]",
                                            unresolvedPlugins));
    }
    return extensions;
  }

  /**
   * Goes over all the dependencies (see {@link MavenProject#getArtifacts()}) that are {@link AbstractPackagePluginMojo#MULE_PLUGIN_CLASSIFIER}
   * while filter them by having a mule-artifact.json descriptor, as there are {@link AbstractPackagePluginMojo#MULE_PLUGIN_CLASSIFIER} that do not have the specific
   * descriptor (such as file-common) by lastly checking they have an {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   *
   * @param project to look for all the dependencies
   * @return a set of {@link MulePluginModel}, not null.
   */
  private Set<MulePluginModel> filterPluginsWithExtensionModelLoader(MavenProject project) {
    return project.getArtifacts().stream()
        .filter(artifact -> MULE_PLUGIN_CLASSIFIER.equals(artifact.getClassifier()) && !SCOPE_TEST.equals(artifact.getScope()))
        .map(this::getDescriptor)
        .filter(Optional::isPresent)
        .map(Optional::get)
        .filter(mulePluginModel -> mulePluginModel.getExtensionModelLoaderDescriptor().isPresent())
        .collect(Collectors.toSet());
  }

  /**
   * Returns a descriptor for any given Mule plugin.
   *
   * @param artifact that should contain the {@link ExtensionPackageMojo#MULE_PLUGIN_JSON_JAR_DESTINATION}
   * @return a {@link MulePluginModel} if exists in the current JAR file, {@link Optional#empty()} otherwise (file-commons use case)
   */
  private Optional<MulePluginModel> getDescriptor(Artifact artifact) {
    Optional<MulePluginModel> result = Optional.empty();
    try {
      final ZipFile pluginJar = new ZipFile(artifact.getFile());
      final ZipEntry mulePluginDescriptorEntry = pluginJar.getEntry(MULE_PLUGIN_JSON_JAR_DESTINATION);
      if (mulePluginDescriptorEntry != null) {
        InputStream stream = pluginJar.getInputStream(mulePluginDescriptorEntry);
        result = Optional.of(new MulePluginModelJsonSerializer().deserialize(IOUtils.toString(stream)));
      }
    } catch (IOException e) {
      throw new IllegalArgumentException(format("Cannot get the file [%s] JAR for the following artifact [%s]",
                                                artifact.getFile().getAbsolutePath(), artifact.toString()));
    }
    return result;
  }

  private ExtensionModel getExtensionModel(ClassLoader pluginClassLoader, Set<ExtensionModel> extensions,
                                           MulePluginModel mulePluginDescriber)
      throws MojoExecutionException, MojoFailureException {
    MuleArtifactLoaderDescriptor loaderDescriptor = getExtensionLoaderDescriptor(mulePluginDescriber);
    log.info(format("Creating ExtensionModel for name:[%s], ID:[%s]", mulePluginDescriber.getName(), loaderDescriptor.getId()));
    final ExtensionModelLoader extensionModelLoader = getExtensionModelLoader(loaderDescriptor.getId(), pluginClassLoader);
    return extensionModelLoader.loadExtensionModel(pluginClassLoader, getDefault(extensions), loaderDescriptor.getAttributes());
  }

  /**
   * @param jsonDescriber an {@link MulePluginModel} instance holding the mule plugin information.
   * @return an {@link MuleArtifactLoaderDescriptor} instance.
   */
  private MuleArtifactLoaderDescriptor getExtensionLoaderDescriptor(MulePluginModel jsonDescriber)
      throws MojoExecutionException, MojoFailureException {
    return jsonDescriber.getExtensionModelLoaderDescriptor()
        .orElseThrow(() -> new MojoExecutionException(
                                                      format("The plugin [%s] does not have a ExtensionLoader descriptor, nothing to generate so far.",
                                                             jsonDescriber.getName())));
  }

  /**
   * Assembles a {@link ClassLoader} from the classpath of the invoker of the current maven's plugin (aka: the connector). This
   * will allow us to consume any {@link ExtensionModelLoader} from it's dependencies.
   *
   * @return a {@link ClassLoader} with the needed elements to find the
   *         {@link ExtensionModelLoader} implementations and then load the
   *         {@link ExtensionModel} properly.
   * @throws MojoFailureException if there are issues while looking for the classpath context or trying to generate the URLs form
   *         each item of the classpath.
   */
  private ClassLoader getInvokerClassLoader(MavenProject project) throws MojoFailureException {
    List<String> invokerClasspathElements;
    try {
      invokerClasspathElements = project.getCompileClasspathElements();
    } catch (DependencyResolutionRequiredException e) {
      throw new MojoFailureException("There was an issue trying to consume the classpath for the current project.", e);
    }
    invokerClasspathElements.add(project.getBuild().getOutputDirectory());
    if (log.isDebugEnabled()) {
      log.debug(format("Classpath to process: [%s]", join(",", invokerClasspathElements)));
    }

    URL urls[] = new URL[invokerClasspathElements.size()];
    for (int i = 0; i < invokerClasspathElements.size(); ++i) {
      final String invokerClasspathElement = invokerClasspathElements.get(i);
      try {
        urls[i] = new File(invokerClasspathElement).toURI().toURL();
      } catch (MalformedURLException e) {
        throw new MojoFailureException(
                                       format("There was an issue trying to convert the element [%s] to an URL. Full classpath: [%s]",
                                              invokerClasspathElement, join(",", invokerClasspathElements)));
      }
    }
    return new URLClassLoader(urls, getClass().getClassLoader());
  }

  /**
   * Given an {@code id} it will look for the first {@link ExtensionModelLoader} implementation, where the acceptance criteria
   * is {@link ExtensionModelLoader#getId()} being equals to it.
   *
   * @param id the ID to look for in each implementation.
   * @param pluginClassLoader {@link ClassLoader} of the current invoker.
   * @return the {@link ExtensionModelLoader} that matches to {@code id}.
   * @throws MojoFailureException if there's no {@link ExtensionModelLoader} that matches to
   *         the parametrized {@code id}.
   */
  private ExtensionModelLoader getExtensionModelLoader(String id, ClassLoader pluginClassLoader)
      throws MojoFailureException {
    List<String> foundIds = new ArrayList<>();
    final ServiceLoader<ExtensionModelLoader> extensionModelLoaders =
        ServiceLoader.load(ExtensionModelLoader.class, pluginClassLoader);
    for (ExtensionModelLoader extensionModelLoader : extensionModelLoaders) {
      foundIds.add(format("Class:[%s]; ID:[%s].", extensionModelLoader.getClass().getName(), extensionModelLoader.getId()));
      if (id.equals(extensionModelLoader.getId())) {
        return extensionModelLoader;
      }
    }
    throw new MojoFailureException(
                                   format("Failure while looking for an implementation class of [%s] class through SPI for the ID [%s]. Found resources: \n%s",
                                          ExtensionModelLoader.class.getName(), id,
                                          join(", \n", foundIds)));
  }
}
