/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation.types;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.mule.extension.maven.documentation.types.ObjectTypeUtils.isMessage;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BinaryType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.RegexType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.util.Reference;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;

public class TypeParserVisitor extends MetadataTypeVisitor {

  Reference<String> holder = new Reference<>();
  private Map<String, List<String>> subtypesIdMap;

  TypeParserVisitor(Map<String, List<String>> subtypesIdMap) {
    this.subtypesIdMap = subtypesIdMap;
  }

  @Override
  protected void defaultVisit(MetadataType metadataType) {
    holder.set("String");
  }

  @Override
  public void visitAnyType(AnyType anyType) {
    holder.set("Any");
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    arrayType.getType().accept(this);
    holder.set("Array of " + holder.get());
  }

  @Override
  public void visitBinaryType(BinaryType binaryType) {
    holder.set("Binary");
  }

  @Override
  public void visitBoolean(BooleanType booleanType) {
    holder.set("Boolean");
  }

  @Override
  public void visitDateTime(DateTimeType dateTimeType) {
    holder.set("DateTime");
  }

  @Override
  public void visitDate(DateType dateType) {
    holder.set("Date");
  }

  @Override
  public void visitNull(NullType nullType) {
    holder.set("Null");
  }

  @Override
  public void visitVoid(VoidType voidType) {
    holder.set("Void");
  }

  @Override
  public void visitNumber(NumberType numberType) {
    holder.set("Number");
  }

  @Override
  public void visitObject(ObjectType objectType) {
    if (isMap(objectType)) {
      holder.set("Object");
      return;
    }

    if (isMessage(objectType)) {
      Iterator<ObjectFieldType> fields = objectType.getFields().iterator();
      // If there are fields then we take the payload and attributes to show in the type section, otherwise is a raw message.
      if (fields.hasNext()) {
        fields.next().getValue().accept(this);
        String payload = this.holder.get();
        fields.next().getValue().accept(this);
        String atts = this.holder.get();
        holder.set(format("Message of [%s] payload and [%s] attributes", payload, atts));
      } else {
        holder.set("Message");
      }
      return;
    }

    Optional<String> objectTypeId = ObjectTypeUtils.getObjectTypeId(objectType);
    if (objectTypeId.isPresent()) {
      String id = objectTypeId.get();
      if (subtypesIdMap.containsKey(id)) {
        StringJoiner builder = new StringJoiner("\n");
        builder.add("One of:\n");
        subtypesIdMap.get(id).forEach(subtype -> builder.add("* <<" + subtype + ">>"));
        holder.set(builder.toString());
      } else {
        holder.set("<<" + id + ">>");
      }
    } else {
      holder.set("Any");
    }
  }

  @Override
  public void visitString(StringType stringType) {
    Optional<EnumAnnotation> enumAnnotation = stringType.getAnnotation(EnumAnnotation.class);
    if (enumAnnotation.isPresent()) {
      StringJoiner builder = new StringJoiner("\n");
      builder.add("Enumeration, one of:\n");
      for (Object value : enumAnnotation.get().getValues()) {
        builder.add("** " + value.toString());
      }
      holder.set(builder.toString());
    } else {
      holder.set("String");
    }
  }

  @Override
  public void visitRegex(RegexType regexType) {
    holder.set("Regex");
  }

  @Override
  public void visitUnion(UnionType unionType) {
    // Creates a list with the possible types.
    List<String> types = unionType.getTypes().stream().map(t -> {
      t.accept(this);
      return this.holder.get();
    }).collect(toList());

    StringJoiner union = new StringJoiner("\n");
    types.forEach(t -> union.add("* " + t));
    holder.set(union.toString());
  }
}
