/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation;

import static java.lang.Character.isLowerCase;
import static java.lang.Character.isUpperCase;
import static java.lang.Character.isWhitespace;
import static java.lang.Character.toUpperCase;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.lang3.StringUtils.isBlank;
import org.mule.runtime.api.meta.DescribedObject;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.HasDisplayModel;

import java.util.Optional;


public class NameUtils {

  public static String prettify(NamedObject component) {
    return getDisplayName((HasDisplayModel) component).orElse(prettify(component.getName()));
  }

  public static String prettify(String name) {
    StringBuilder pretty = new StringBuilder();
    boolean shouldCapitalizeNext = true;
    for (int i = 0; i < name.length(); i++) {
      char c = name.charAt(i);

      if (c == '-' || isWhitespace(c)) {
        shouldCapitalizeNext = true;
        pretty.append(" ");
        continue;
      }

      if (shouldAddWhitespace(name, i, c) && !endsWithWhitespace(pretty)) {
        pretty.append(" ");
      }

      if (shouldCapitalizeNext) {
        pretty.append(toUpperCase(c));
        shouldCapitalizeNext = false;
      } else {
        pretty.append(c);
      }
    }
    return pretty.toString().trim();
  }

  private static Optional<String> getDisplayName(HasDisplayModel model) {
    Optional<DisplayModel> displayModel = model.getDisplayModel();
    return displayModel.isPresent() && !isBlank(displayModel.get().getDisplayName())
        ? Optional.ofNullable(displayModel.get().getDisplayName()) : empty();
  }

  public static Optional<String> getDescription(HasDisplayModel model, DescribedObject describedObject) {
    Optional<String> optionalDescription = empty();

    String description = describedObject.getDescription();
    if (!isBlank(description)) {
      optionalDescription = of(description);
    } else {
      Optional<DisplayModel> displayModel = model.getDisplayModel();
      if (displayModel.isPresent() && !isBlank(displayModel.get().getSummary())) {
        optionalDescription = of(displayModel.get().getSummary());
      }
    }

    return optionalDescription;
  }

  private static boolean shouldAddWhitespace(String name, int i, char c) {
    return isUpperCase(c) && (!previousIsUppercase(name, i) || (previousIsUppercase(name, i) && nextIsLowercase(name, i)));
  }

  private static boolean previousIsUppercase(String name, int i) {
    return i > 0 && isUpperCase(name.charAt(i - 1));
  }

  private static boolean nextIsLowercase(String name, int i) {
    return i < name.length() - 1 && isLowerCase(name.charAt(i + 1));
  }

  private static boolean endsWithWhitespace(StringBuilder builder) {
    return builder.length() > 0 && isWhitespace(builder.charAt(builder.length() - 1));
  }
}
