package org.mule.extension.maven;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.notExists;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.listFiles;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.maven.plugins.annotations.LifecyclePhase.COMPILE;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.io.filefilter.FileFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Mojo responsible of looking for an existing {@link ExtensionPackageMojo#MULE_ARTIFACT_JSON} in META-INF folder.
 * <p/>
 * If it doesn't exists, it assumes it's a <module/> scenario, to which it will try to generate one with the following
 * conventions:
 * <ol>
 * <li>Will scan the /classes looking for anything that starts with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends with
 * {@link #SUFFIX_SMART_CONNECTOR_NAME}</li>
 * <li>For each element found, it will open it and look if they are valid XML files and their root elements matches to
 * {@link #MODULE_ROOT_ELEMENT}</li>
 * <li>There must be one, and only one, file in the /classes that match the previous rules. If not, fails.</li>
 * </ol>
 *
 * @since 1.0
 */
@Mojo(name = "extension-descriptor", defaultPhase = COMPILE, threadSafe = true)
public class ExtensionDescriptorMojo extends AbstractMuleMojo {

  private static final String META_INF = "META-INF";
  private static final String TEMPORARY_MULE_ARTIFACT_JSON = "temporal-" + MULE_ARTIFACT_JSON;
  private static final String TEMPORARY_COMPILE_MULE_ARTIFACT_JSON = "temporal-compile-" + MULE_ARTIFACT_JSON;

  private static final String MULE_LOADER_ID = "mule";
  public static final String XML_BASED_EXTENSION_MODEL_LOADER = "xml-based";
  public static final String RESOURCE_XML = "resource-xml";
  private static final String VALIDATE_XML = "validate-xml";
  private static final String PREFIX_SMART_CONNECTOR_NAME = "module-";
  private static final String SUFFIX_SMART_CONNECTOR_NAME = ".xml";
  private static final String MODULE_ROOT_ELEMENT = "module";
  private static final String NAME_ATTRIBUTE = "name";
  private static final String MIN_MULE_VERSION_ATTRIBUTE = "minMuleVersion";

  /**
   * Returns a {@link Path} to an existing descriptor file or fails.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor file
   * @throws MojoFailureException if the descriptor file is absent (probably because this Mojo hasn't been executed)
   */
  public static Path descriptorPathOrFail(File outputDirectory) throws MojoFailureException {
    return descriptorPath(outputDirectory, TEMPORARY_MULE_ARTIFACT_JSON);
  }

  /**
   * Returns a {@link Path} to an existing descriptor compile file (validates XML) or fails.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor compile file
   * @throws MojoFailureException if the descriptor file is absent (probably because this Mojo hasn't been executed)
   */
  public static Path descriptorCompilePathOrFail(File outputDirectory) throws MojoFailureException {
    return descriptorPath(outputDirectory, TEMPORARY_COMPILE_MULE_ARTIFACT_JSON);
  }

  private static Path descriptorPath(File outputDirectory, String descriptorLocation) throws MojoFailureException {
    final Path path = Paths.get(outputDirectory.getAbsolutePath(), descriptorLocation);
    if (notExists(path)) {
      throw new MojoFailureException(format("Should not have reach this point, could not obtain descriptor file from [%s]",
                                            path));
    }
    return path;
  }

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    // This file may be created by MulePluginDescriptorGenerator
    final Path originalDescriptorPath = Paths.get(project.getBuild().getOutputDirectory(), META_INF, MULE_ARTIFACT_JSON);
    final Path descriptorPath = Paths.get(outputDirectory.getAbsolutePath(), TEMPORARY_MULE_ARTIFACT_JSON);
    if (exists(originalDescriptorPath)) {
      if (descriptorPath.toFile().exists()) {
        deleteQuietly(descriptorPath.toFile());
      }
      try {
        Files.copy(originalDescriptorPath, descriptorPath);
      } catch (IOException e) {
        throw new MojoExecutionException(format("There was an issue copying the descriptor file from [%s] to [%s]",
                                                originalDescriptorPath,
                                                descriptorPath),
                                         e);
      }
    } else {
      final Path descriptorCompilePath = Paths.get(outputDirectory.getAbsolutePath(), TEMPORARY_COMPILE_MULE_ARTIFACT_JSON);
      getLog().info(format("No [%s] descriptor found, trying to create one", originalDescriptorPath));
      createDescriptor(descriptorPath, descriptorCompilePath);
    }
  }

  /**
   * It assumes there will be one, and only one,
   * 
   * <pre>
   * mule - fillWithAName.xml
   * </pre>
   * 
   * module in the current working directory.
   *
   * @param descriptorPath final path where the descriptor must be before being either consumed or packaged. See
   *        {@link #descriptorPathOrFail(File)}
   * @param descriptorCompilePath final path where the compile descriptor must be when validating the XML of the connector.
   * @throws MojoExecutionException if there aren't any module to read, or if the module misses the {@link #NAME_ATTRIBUTE} or
   *         {@link #MIN_MULE_VERSION_ATTRIBUTE} attributes
   */
  private void createDescriptor(Path descriptorPath, Path descriptorCompilePath) throws MojoExecutionException {
    final String baseDirectory = project.getBuild().getOutputDirectory();
    File moduleFile = getModuleFile(baseDirectory);
    getLog().info(format("Generating [%s] descriptor for the <module> found in [%s]", MULE_ARTIFACT_JSON,
                         moduleFile.getAbsolutePath()));

    final Document doc = getModule(moduleFile).get();
    final String name = doc.getDocumentElement().getAttribute(NAME_ATTRIBUTE);
    if (isBlank(name)) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorPath));
    }
    String minMuleVersion = doc.getDocumentElement().getAttribute(MIN_MULE_VERSION_ATTRIBUTE);
    if (isBlank(minMuleVersion)) {
      throw new MojoExecutionException(format("The module being read [%s] must have a non-empty minMuleVersion attribute",
                                              moduleFile));
    }

    final String relativeModuleFileName =
        moduleFile.getAbsolutePath().substring(baseDirectory.length() + 1,
                                               moduleFile.getAbsolutePath().length());
    createDescriptor(descriptorPath, name, minMuleVersion, relativeModuleFileName, false);
    createDescriptor(descriptorCompilePath, name, minMuleVersion, relativeModuleFileName, true);
  }

  private void createDescriptor(Path descriptorPath, String name, String minMuleVersion, String resourceXml, boolean validateXml)
      throws MojoExecutionException {
    final MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder()
        .setName(name)
        .setMinMuleVersion(minMuleVersion);
    mulePluginModelBuilder.withBundleDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()));
    mulePluginModelBuilder.withClassLoaderModelDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()));
    mulePluginModelBuilder.withExtensionModelDescriber().setId(XML_BASED_EXTENSION_MODEL_LOADER)
        .addProperty(RESOURCE_XML, resourceXml)
        .addProperty(VALIDATE_XML, validateXml);

    final MulePluginModel build = mulePluginModelBuilder.build();
    final String descriptor = new MulePluginModelJsonSerializer().serialize(build);
    try {
      Files.write(descriptorPath, descriptor.getBytes(), StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorPath),
                                       e);
    }
  }

  /**
   * Looks in {@code baseDirectory} recursively looking for files that start with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends
   * with {@link #SUFFIX_SMART_CONNECTOR_NAME} to then validate through {@link #getModule(File)} if they are a <module/> or not.
   *
   * @param baseDirectory base path to start looking for <module/>
   * @return a {@link File} that targets the only <module/> of the current project.
   * @throws MojoExecutionException if the amount of <module/>s found is different than 1. There can be only one, Highlander.
   */
  private File getModuleFile(String baseDirectory) throws MojoExecutionException {
    final List<File> modulesFiles = listFiles(new File(baseDirectory), new FileFileFilter() {

      @Override
      public boolean accept(File file) {
        return file.getName().startsWith(PREFIX_SMART_CONNECTOR_NAME)
            && file.getName().endsWith(SUFFIX_SMART_CONNECTOR_NAME);
      }
    }, TrueFileFilter.INSTANCE)
        .stream()
        .filter(file -> getModule(file).isPresent())
        .collect(Collectors.toList());

    if (modulesFiles.size() > 1) {
      final String xmlModules = join("\n,", modulesFiles.stream().map(File::getAbsolutePath).collect(Collectors.toList()));
      throw new MojoExecutionException(format("There are several XML files that have a <module> root element, when there must be only one. Files with <module> as root element are: [%s]",
                                              xmlModules));
    } else if (modulesFiles.isEmpty()) {
      throw new MojoExecutionException(format("There's no XML files that has a <module> root element, thus is impossible to auto generate a [%s] descriptor file. The file must start with [%s] and end with [%s], such as [%s]",
                                              MULE_ARTIFACT_JSON,
                                              PREFIX_SMART_CONNECTOR_NAME,
                                              SUFFIX_SMART_CONNECTOR_NAME,
                                              PREFIX_SMART_CONNECTOR_NAME + "foo" + SUFFIX_SMART_CONNECTOR_NAME));
    }
    return modulesFiles.get(0);
  }

  /**
   * Given an existing {@link File}, it returns a {@link Document} if it's able to parse it and the root element's name matches
   * with {@link #MODULE_ROOT_ELEMENT}.
   *
   * @param file to be read
   * @return the parsed file if it's a <module/>
   */
  private Optional<Document> getModule(File file) {
    Optional<Document> result = empty();
    DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    try {
      DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
      Document doc = dBuilder.parse(file);
      if (doc.getDocumentElement().getNodeName().equals(MODULE_ROOT_ELEMENT)) {
        result = of(doc);
      }
    } catch (ParserConfigurationException | SAXException | IOException e) {
      // If it fails, then the file wasn't a <module> after all :)
    }
    return result;
  }
}
