/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.plugin.maven;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PACKAGE;
import static org.apache.maven.plugins.annotations.ResolutionScope.RUNTIME;
import static org.mule.plugin.maven.PluginProperties.ARTIFACT_EXPORT_CLASS_PACKAGES;
import static org.mule.plugin.maven.PluginProperties.ARTIFACT_EXPORT_RESOURCES;
import static org.mule.plugin.maven.PluginProperties.PLUGIN_BUNDLE;
import static org.mule.plugin.maven.ServiceProperties.SERVICE_PROVIDER_CLASS_NAME;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Properties;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;


/**
 * Build a Mule service archive.
 */
@Mojo(name = "package-service", defaultPhase = PACKAGE, requiresDependencyResolution = RUNTIME, threadSafe = true)
public class ServicePackageMojo extends AbstractPackagePluginMojo {

  @Parameter(defaultValue = "${project.basedir}", readonly = true)
  private File baseDir;

  /**
   * If provided, it will be out in the final archive. If not, it will be generated based on the extension manifest.
   */
  @Parameter(defaultValue = SERVICE_PROPERTIES_FILE_NAME)
  private File customServiceProperties;

  @Override
  protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException {
    addServiceJar(archiver);
    super.addToArchiver(archiver);
  }

  private void addServiceJar(ModuleArchiver archiver) throws MojoFailureException {
    String jarName = finalName + ".jar";
    File jar = new File(outputDirectory, jarName);
    if (!jar.exists()) {
      throw new MojoFailureException(format("Artifact jar <%1s> doesn't exist in target folder", jarName));
    }
    archiver.addLib(jar);
  }

  @Override
  protected void doAddArtifactProperties(File servicePropertiesFile) throws MojoFailureException {
    try {
      InputStream is = new FileInputStream(getServicePropertiesFile());
      Properties loadProps = new Properties();
      loadProps.load(is);
      OutputStream os = new FileOutputStream(servicePropertiesFile);
      PluginProperties serviceProperties = new ServiceProperties(loadProps.getProperty(SERVICE_PROVIDER_CLASS_NAME),
                                                                 loadProps.getProperty(ARTIFACT_EXPORT_CLASS_PACKAGES),
                                                                 loadProps.getProperty(ARTIFACT_EXPORT_RESOURCES),
                                                                 join(getPluginDependencies(), ","),
                                                                 loadProps.getProperty(PLUGIN_BUNDLE));
      serviceProperties.writeTo(os);
    } catch (IOException e) {
      throw new MojoFailureException("Error copying properties file " + e.getMessage());
    }
  }

  @Override
  protected void addArtifactProperties(ModuleArchiver moduleArchiver) throws MojoFailureException {
    File pluginPropertiesFile = new File(outputDirectory, SERVICE_PROPERTIES_FILE_NAME);
    doAddArtifactProperties(pluginPropertiesFile);
    moduleArchiver.add(pluginPropertiesFile);
  }

  public File getServicePropertiesFile() {
    if (customServiceProperties.exists()) {
      return customServiceProperties;
    }
    return new File(baseDir, "src/main/resources/" + SERVICE_PROPERTIES_FILE_NAME);
  }

}
