/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracer.exporter.impl.optel;

import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.OPEN_TELEMETRY_EXPORTER;

import static java.util.Collections.emptySet;

import static io.opentelemetry.sdk.trace.samplers.Sampler.alwaysOff;
import static io.opentelemetry.sdk.trace.samplers.Sampler.alwaysOn;
import static io.opentelemetry.sdk.trace.samplers.Sampler.parentBasedBuilder;
import static io.opentelemetry.sdk.trace.samplers.Sampler.traceIdRatioBased;
import static org.hamcrest.CoreMatchers.endsWith;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.mock;

import io.qameta.allure.Issue;
import org.mule.runtime.api.profiling.tracing.Span;
import org.mule.runtime.tracer.api.span.info.InitialExportInfo;
import org.mule.runtime.tracer.api.span.info.InitialSpanInfo;
import org.mule.runtime.tracer.exporter.impl.OpenTelemetrySpanExporter;

import java.util.HashMap;
import java.util.Map;

import io.opentelemetry.sdk.resources.Resource;
import io.opentelemetry.sdk.trace.SpanProcessor;
import io.opentelemetry.sdk.trace.samplers.Sampler;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.Test;

@Feature(PROFILING)
@Story(OPEN_TELEMETRY_EXPORTER)
@RunWith(Parameterized.class)
public class OpenTelemetrySpanExporterTestCase {

  private final InitialSpanInfo initialSpanInfo;
  private final Sampler sampler;
  private final Map<String, String> parentSpanContext;
  private final String expectedTraceFlags;

  @Parameterized.Parameters(name = "Parent SpanContext: {2} - Sampler: {1}")
  public static Object[][] params() {
    return new Object[][] {
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "00"},
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "00"},
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331"), "00"},
        {getExportableInitialSpanInfo(), getAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "00"},
        {getExportableInitialSpanInfo(), getAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "00"},
        {getExportableInitialSpanInfo(), getAlwaysOffSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331"), "00"},
        {getExportableInitialSpanInfo(), getAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "01"},
        {getExportableInitialSpanInfo(), getAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getAlwaysOnSampler(),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(1),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "00"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(1),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(1),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(0),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-00"), "00"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(0),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01"), "01"},
        {getExportableInitialSpanInfo(), getParentBasedTraceIdRatioSampler(0),
            getParentSpanContext("00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331"), "00"}
    };
  }

  private static Map<String, String> getParentSpanContext(String traceParent) {
    Map<String, String> parentSpanContext = new HashMap<>(2);
    parentSpanContext.put("traceparent", traceParent);
    return parentSpanContext;
  }

  public OpenTelemetrySpanExporterTestCase(InitialSpanInfo initialSpanInfo, Sampler sampler,
                                           Map<String, String> parentSpanContext, String expectedTraceFlags) {
    this.initialSpanInfo = initialSpanInfo;
    this.sampler = sampler;
    this.parentSpanContext = parentSpanContext;
    this.expectedTraceFlags = expectedTraceFlags;
  }

  @Test
  @Issue("W-19784910")
  public void samplingDecisionMustBeReflectedAtTraceFlags() {
    OpenTelemetrySpanExporter openTelemetrySpanExporter =
        new OpenTelemetrySpanExporter(mock(Span.class), initialSpanInfo, "testArtifactId", "testArtifactType",
                                      mock(SpanProcessor.class), false, mock(Resource.class), sampler);
    openTelemetrySpanExporter.updateParentSpanFrom(parentSpanContext);
    Map<String, String> exportedSpanAsMap = openTelemetrySpanExporter.exportedSpanAsMap();
    assertThat(exportedSpanAsMap.get("traceparent"), endsWith(expectedTraceFlags));
  }

  private static Sampler getParentBasedAlwaysOnSampler() {
    return parentBasedBuilder(alwaysOn()).build();
  }

  private static Sampler getParentBasedAlwaysOffSampler() {
    return parentBasedBuilder(alwaysOff()).build();
  }

  private static Object getParentBasedTraceIdRatioSampler(double ratio) {
    return parentBasedBuilder(traceIdRatioBased(ratio)).build();
  }

  private static Object getAlwaysOnSampler() {
    return alwaysOn();
  }

  private static Sampler getAlwaysOffSampler() {
    return alwaysOff();
  }

  private static InitialSpanInfo getExportableInitialSpanInfo() {
    InitialSpanInfo initialSpanInfo = mock(InitialSpanInfo.class);
    InitialExportInfo initialExportInfo = mock(InitialExportInfo.class);
    when(initialSpanInfo.getInitialExportInfo()).thenReturn(initialExportInfo);
    when(initialExportInfo.isExportable()).thenReturn(true);
    when(initialExportInfo.noExportUntil()).thenReturn(emptySet());
    return initialSpanInfo;
  }

}
