/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracer.exporter.config.impl;

import static org.mule.runtime.api.util.IOUtils.getResourceAsUrl;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher.MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_CA_FILE_LOCATION;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_KEY_FILE_LOCATION;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_SERVICE_NAME;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_SERVICE_NAMESPACE;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER_ARG;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH;
import static org.mule.tck.probe.PollingProber.DEFAULT_POLLING_INTERVAL;
import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.DEFAULT_CORE_EVENT_TRACER;

import static java.io.File.createTempFile;
import static java.lang.Boolean.TRUE;
import static java.lang.System.getProperty;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Paths.get;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.Objects.requireNonNull;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.config.internal.model.dsl.config.PropertyNotFoundException;
import org.mule.runtime.core.api.MuleContext;
import org.mule.tck.junit5.SystemProperty;
import org.mule.tck.junit5.SystemPropertyExtension;
import org.mule.tck.probe.JUnitLambdaProbe;
import org.mule.tck.probe.PollingProber;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URI;
import java.nio.file.Path;
import java.util.Properties;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.params.ParameterizedClass;
import org.junit.jupiter.params.provider.CsvSource;

@Feature(PROFILING)
@Story(DEFAULT_CORE_EVENT_TRACER)
@ParameterizedClass(name = "[{index}] {arguments}")
@CsvSource(useHeadersInDisplayName = true, value = {
    "enableConfigInFile, valuePropertyNonSystemPropertyConfDirectory, valuePropertySystemProperty",
    "true, valueNonSystemProperty, valueSystemProperty",
    "false, valueNonSystemPropertyConfDirectory, valueSystemPropertyConfDirectory"
})
@SystemProperty(name = MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY, value = "1000")
public class FileSpanExporterConfigurationTestCase {

  public static final String TEST_NOT_FOUND_CONF_FILE_NAME = "test-not-found.conf";
  public static final String TEST_CONF_FILE_NAME = "test.conf";
  public static final String TEST_EXTERNAL_CONF_PATH = "conf/test.conf";
  public static final String SYSTEM_PROPERTY_VALUE = "system_property_value";
  public static final String TEST_ORIGINAL_CONF_FILE = "test-original.conf";
  private static final long TIMEOUT_MILLIS = 10000l;
  public static final String TEST_OVERWRITTEN_CONF_FILE = "test-overwritten.conf";

  public static final String KEY_PROPERTY_NON_SYSTEM_PROPERTY = "keyNonSystemProperty";
  public static final String KEY_PROPERTY_SYSTEM_PROPERTY = "keySystemProperty";
  public static final String KEY_UNSET_SYSTEM_PROPERTY = "keyUnsetSystemProperty";
  public static final String KEY_BOOLEAN_VALUE = "booleanValue";
  public static final String KEY_STRING_VALUE_ONE = "stringValueOne";
  public static final String KEY_STRING_VALUE_TWO = "stringValueTwo";
  public static final String NO_KEY_IN_FILE = "noKeyInFile";
  public static final String NON_ARTIFACT_LEVEL_PROPERTY = "nonArtifactLevelProperty";
  public static final String DEPLOYMENT_PROPERTY_VALUE = "deployment_property_value";
  public static final String OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY = "overWrittenArtifactLevelProperty";

  private final String valueNonSystemProperty;
  private final String configuredSystemPropertyKey;

  private MuleContext muleContext;

  @RegisterExtension
  public final SystemPropertyExtension enableConfigInFileProperty;

  @RegisterExtension
  public final SystemPropertyExtension tracingConfigurationFilePath;

  @RegisterExtension
  public final SystemPropertyExtension configuredSystemProperty;

  @RegisterExtension
  public final SystemPropertyExtension nonArtifactLevelSystemProperty;

  @RegisterExtension
  public final SystemPropertyExtension overWrittenArtifactLevelSystemProperty;

  public FileSpanExporterConfigurationTestCase(String enableConfigInFile,
                                               String valuePropertyNonSystemPropertyConfDirectory,
                                               String valuePropertySystemProperty)
      throws IOException {
    this.valueNonSystemProperty = valuePropertyNonSystemPropertyConfDirectory;
    this.configuredSystemPropertyKey = valuePropertySystemProperty;

    Path externalConfigurationPath = generateExternalConfigurationPath();

    this.enableConfigInFileProperty =
        new SystemPropertyExtension(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY, enableConfigInFile);
    this.tracingConfigurationFilePath =
        new SystemPropertyExtension(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH,
                                    externalConfigurationPath.toAbsolutePath().toString());
    this.configuredSystemProperty =
        new SystemPropertyExtension(configuredSystemPropertyKey, SYSTEM_PROPERTY_VALUE);

    this.nonArtifactLevelSystemProperty = new SystemPropertyExtension(NON_ARTIFACT_LEVEL_PROPERTY, SYSTEM_PROPERTY_VALUE);

    this.overWrittenArtifactLevelSystemProperty =
        new SystemPropertyExtension(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY, SYSTEM_PROPERTY_VALUE);
  }

  @BeforeEach
  void setUp() {
    muleContext = mock(MuleContext.class);
    // Deployment properties that can be read by the tracing configuration.
    Properties deploymentPropertiesStub = new Properties();
    deploymentPropertiesStub.put(MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER_ARG, "1");
    deploymentPropertiesStub.put(MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER, "always_on");
    deploymentPropertiesStub.put("mule.openTelemetry.exporter.resource.service.namespace", "deploymentServiceNamespace");
    deploymentPropertiesStub.put("mule.openTelemetry.exporter.resource.service.name", "deploymentServiceName");
    deploymentPropertiesStub.put(NON_ARTIFACT_LEVEL_PROPERTY, DEPLOYMENT_PROPERTY_VALUE);
    deploymentPropertiesStub.put(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY, DEPLOYMENT_PROPERTY_VALUE);
    when(muleContext.getDeploymentProperties()).thenReturn(deploymentPropertiesStub);
    // The execution classloader (which represents the artifact classloader) will be the classloader of the test.
    when(muleContext.getExecutionClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());
  }

  /**
   * This temporary directory simulates a manual configuration of the config file path. It contains the configuration file to be
   * read in the case of a manual directory configuration.
   *
   * @return Manually configured configuration file path.
   * @throws IOException When the temporary path/file cannot be created.
   */
  private Path generateExternalConfigurationPath() throws IOException {
    Path externalSpanExporterConfigurationFile = createTempDirectory(this.getClass().getName());
    externalSpanExporterConfigurationFile.toFile().deleteOnExit();
    // This temporary file simulates a manual configuration of the config file path and name.
    Path externalConfigFilePath = externalSpanExporterConfigurationFile.resolve(TEST_CONF_FILE_NAME);
    copy(requireNonNull(getClass().getClassLoader().getResourceAsStream(TEST_EXTERNAL_CONF_PATH)), externalConfigFilePath,
         REPLACE_EXISTING);
    externalConfigFilePath.toFile().deleteOnExit();
    return externalSpanExporterConfigurationFile;
  }

  @Test
  void returnsTheValueForANonSystemProperty() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), equalTo(
                                                                                                       valueNonSystemProperty));
  }

  @Test
  void returnsTheResolvedSystemProperty() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), equalTo(
                                                                                                   SYSTEM_PROPERTY_VALUE));
  }

  @Test
  void whenASystemPropertyCannotBeResolvedAnExceptionIsRaised() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThrows(PropertyNotFoundException.class,
                 () -> fileSpanExporterConfiguration.getStringValue(KEY_UNSET_SYSTEM_PROPERTY));
  }

  @Test
  void whenNoPropertyIsInTheFileNullValueIsReturned() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(NO_KEY_IN_FILE), is(nullValue()));
  }

  @Test
  void whenFileIsNotFoundNoPropertyIsFound() {
    FileSpanExporterConfiguration testNoFileFoundSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_NOT_FOUND_CONF_FILE_NAME);
    assertThat(testNoFileFoundSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), is(nullValue()));
    assertThat(testNoFileFoundSpanExporterConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), is(nullValue()));
  }

  @Test
  void readsStringValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    String stringValueOne = testFileSpanExporterConfiguration.getStringValue(KEY_STRING_VALUE_ONE);
    String stringValueTwo = testFileSpanExporterConfiguration.getStringValue(KEY_STRING_VALUE_TWO);

    assertThat(stringValueOne, is("stringValueOne"));
    assertThat(stringValueTwo, is("stringValueTwo"));
  }

  @Test
  void readsBooleanValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    String booleanValue = testFileSpanExporterConfiguration.getStringValue(KEY_BOOLEAN_VALUE);

    assertThat(booleanValue, is("true"));
  }

  @Test
  void whenValueCorrespondingToPathGetAbsoluteValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    Path caFileLocationPath = testFileSpanExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_EXPORTER_CA_FILE_LOCATION);
    Path keyFileLocationPath = testFileSpanExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_EXPORTER_KEY_FILE_LOCATION);

    assertThat(caFileLocationPath, is(notNullValue()));
    assertThat(keyFileLocationPath, is(notNullValue()));

    assertThat(caFileLocationPath.isAbsolute(), is(TRUE));
    assertThat(keyFileLocationPath.isAbsolute(), is(TRUE));
  }

  @Test
  @Issue("W-20139161")
  void artifactLevelPropertiesMustBeReadFromDeploymentProperties() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);

    assertThat(testFileSpanExporterConfiguration.getStringValue(MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER), is("always_on"));
    assertThat(testFileSpanExporterConfiguration.getStringValue(MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER_ARG), is("1"));
    assertThat(testFileSpanExporterConfiguration.getStringValue(MULE_OPEN_TELEMETRY_EXPORTER_SERVICE_NAMESPACE),
               is("deploymentServiceNamespace"));
    assertThat(testFileSpanExporterConfiguration.getStringValue(MULE_OPEN_TELEMETRY_EXPORTER_SERVICE_NAME),
               is("deploymentServiceName"));
  }

  @Test
  @Issue("W-20139161")
  void nonArtifactLevelPropertiesMustNotBeReadFromDeploymentProperties() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(testFileSpanExporterConfiguration.getStringValue(NON_ARTIFACT_LEVEL_PROPERTY), is(SYSTEM_PROPERTY_VALUE));
  }

  @Test
  @Issue("W-20139161")
  void overWrittenArtifactLevelPropertiesMustBeReadFromDeploymentProperties() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(getProperty(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY), is(SYSTEM_PROPERTY_VALUE));
    assertThat(testFileSpanExporterConfiguration.getStringValue(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY),
               is(DEPLOYMENT_PROPERTY_VALUE));
  }

  @Test
  @Disabled("To be fixed in W-16676258")
  void configurationFileChanged() throws Exception {
    File file = createTempFile("tracing", "test");
    Path testFile = get(file.getPath());
    URI originalConfigFileUri = getResourceAsUrl(TEST_ORIGINAL_CONF_FILE, getClass()).toURI();
    URI overwrittenConfigFileUri = getResourceAsUrl(TEST_OVERWRITTEN_CONF_FILE, getClass()).toURI();
    copy(get(originalConfigFileUri), testFile, REPLACE_EXISTING);

    final TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, file.getAbsolutePath());

    testFileSpanExporterConfiguration.doOnConfigurationChanged(() -> {
      testFileSpanExporterConfiguration.initialise();
      testFileSpanExporterConfiguration.changed = true;
    });
    assertThat(testFileSpanExporterConfiguration.getStringValue("key"), equalTo("value"));
    copy(get(overwrittenConfigFileUri), testFile, REPLACE_EXISTING);
    new PollingProber(TIMEOUT_MILLIS, DEFAULT_POLLING_INTERVAL)
        .check(new JUnitLambdaProbe(() -> testFileSpanExporterConfiguration.changed));
    assertThat(testFileSpanExporterConfiguration.getStringValue("key"), equalTo("value-overwritten"));
  }

  @Test
  @Issue("W-13521171")
  void disposeInterruptsFileWatcherThread() throws Exception {
    File file = createTempFile("tracing", "test");
    file.deleteOnExit();

    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, file.getAbsolutePath());

    // Initialise the configuration
    testFileSpanExporterConfiguration.initialise();
    Thread watcherThread = testFileSpanExporterConfiguration.getFileWatcherThread();
    assertThat("Thread should be alive after initialise",
               watcherThread != null && watcherThread.isAlive(), is(true));

    // Dispose the configuration
    testFileSpanExporterConfiguration.dispose();
    new PollingProber(TIMEOUT_MILLIS, DEFAULT_POLLING_INTERVAL)
        .check(new JUnitLambdaProbe(() -> {
          Thread fileWatcherThread = testFileSpanExporterConfiguration.getFileWatcherThread();
          return !fileWatcherThread.isAlive();
        }));
    watcherThread = testFileSpanExporterConfiguration.getFileWatcherThread();
    assertThat("Thread should not be null or alive after dispose",
               watcherThread != null && !watcherThread.isAlive(), is(true));
  }

  /**
   * {@link FileSpanExporterConfiguration} used for testing properties file.
   */
  private static class TestFileSpanExporterConfiguration extends FileSpanExporterConfiguration {

    private final String confFileName;
    private boolean changed;

    public TestFileSpanExporterConfiguration(MuleContext muleContext, String confFileName) {
      super(muleContext);
      this.confFileName = confFileName;
    }

    @Override
    protected String getSignalConfigurationFileName() {
      return confFileName;
    }

    @Override
    protected boolean isArtifactLevelProperty(String configurationKey) {
      return super.isArtifactLevelProperty(configurationKey) || configurationKey.equals(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY);
    }

    /**
     * Exposes the file watcher thread for testing purposes. Returns null if the thread was never created or has been disposed.
     */
    private Thread getFileWatcherThread() {
      try {
        Field field = FileSpanExporterConfiguration.class.getDeclaredField("tracingConfigurationFileWatcher");
        field.setAccessible(true);
        Object watcher = field.get(this);
        return watcher != null ? (Thread) watcher : null;
      } catch (Exception e) {
        // Return null if reflection fails or field doesn't exist
        return null;
      }
    }
  }
}
