/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracer.exporter.config.impl;

import static org.mule.runtime.api.util.IOUtils.getResourceAsUrl;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher.MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_CA_FILE_LOCATION;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_KEY_FILE_LOCATION;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH;
import static org.mule.tck.probe.PollingProber.DEFAULT_POLLING_INTERVAL;
import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.DEFAULT_CORE_EVENT_TRACER;

import static java.io.File.createTempFile;
import static java.lang.Boolean.TRUE;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Paths.get;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.Objects.requireNonNull;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.config.internal.model.dsl.config.PropertyNotFoundException;
import org.mule.runtime.core.api.MuleContext;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.tck.probe.JUnitLambdaProbe;
import org.mule.tck.probe.PollingProber;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@Feature(PROFILING)
@Story(DEFAULT_CORE_EVENT_TRACER)
@RunWith(Parameterized.class)
public class FileSpanExporterConfigurationTestCase {

  public static final String TEST_NOT_FOUND_CONF_FILE_NAME = "test-not-found.conf";
  public static final String TEST_CONF_FILE_NAME = "test.conf";
  public static final String TEST_EXTERNAL_CONF_PATH = "conf/test.conf";
  public static final String SYSTEM_PROPERTY_VALUE = "system_property_value";
  public static final String TEST_ORIGINAL_CONF_FILE = "test-original.conf";
  private static final long TIMEOUT_MILLIS = 10000l;
  public static final String TEST_OVERWRITTEN_CONF_FILE = "test-overwritten.conf";
  private final String valueNonSystemProperty;

  @Parameterized.Parameters(name = "enableTracingConfigurationAtApplicationLevel: {0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {{true, "valueNonSystemProperty", "valueSystemProperty"},
        {false, "valueNonSystemPropertyConfDirectory", "valueSystemPropertyConfDirectory"}});
  }

  @Rule
  public SystemProperty enableTracingConfigurationAtApplicationLevel;

  @Rule
  public SystemProperty externalConfigurationFilePath;

  @Rule
  public SystemProperty configurationFileWatcherDelay =
      new SystemProperty(MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY, "1000");

  public FileSpanExporterConfigurationTestCase(boolean enableConfigInFile,
                                               String valuePropertyNonSystemPropertyConfDirectory,
                                               String valuePropertySystemProperty)
      throws IOException {
    this.valueNonSystemProperty = valuePropertyNonSystemPropertyConfDirectory;
    configuredSystemProperty = new SystemProperty(valuePropertySystemProperty, SYSTEM_PROPERTY_VALUE);
    enableTracingConfigurationAtApplicationLevel =
        new SystemProperty(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY,
                           Boolean.toString(enableConfigInFile));
    externalConfigurationFilePath = new SystemProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH,
                                                       generateExternalConfigurationPath().toAbsolutePath().toString());
  }

  public static final String KEY_PROPERTY_NON_SYSTEM_PROPERTY = "keyNonSystemProperty";
  public static final String KEY_PROPERTY_SYSTEM_PROPERTY = "keySystemProperty";
  public static final String KEY_UNSET_SYSTEM_PROPERTY = "keyUnsetSystemProperty";
  public static final String KEY_BOOLEAN_VALUE = "booleanValue";
  public static final String KEY_STRING_VALUE_ONE = "stringValueOne";
  public static final String KEY_STRING_VALUE_TWO = "stringValueTwo";
  public static final String NO_KEY_IN_FILE = "noKeyInFile";
  private MuleContext muleContext;

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Rule
  public ExpectedException expected = ExpectedException.none();

  @Rule
  public SystemProperty configuredSystemProperty;

  @Before
  public void setUp() throws IOException {
    muleContext = mock(MuleContext.class);
    when(muleContext.getExecutionClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());
  }

  /**
   * This temporary directory simulates a manual configuration of the config file path. It contains the configuration file to be
   * read in the case of a manual directory configuration.
   *
   * @return Manually configured configuration file path.
   * @throws IOException When the temporary path/file cannot be created.
   */
  private Path generateExternalConfigurationPath() throws IOException {
    Path externalSpanExporterConfigurationFile = createTempDirectory(this.getClass().getName());
    externalSpanExporterConfigurationFile.toFile().deleteOnExit();
    // This temporary file simulates a manual configuration of the config file path and name.
    Path externalConfigFilePath = externalSpanExporterConfigurationFile.resolve(TEST_CONF_FILE_NAME);
    copy(requireNonNull(getClass().getClassLoader().getResourceAsStream(TEST_EXTERNAL_CONF_PATH)), externalConfigFilePath,
         REPLACE_EXISTING);
    externalConfigFilePath.toFile().deleteOnExit();
    return externalSpanExporterConfigurationFile;
  }


  @Test
  public void returnsTheValueForANonSystemProperty() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), equalTo(
                                                                                                       valueNonSystemProperty));
  }

  @Test
  public void returnsTheResolvedSystemProperty() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), equalTo(
                                                                                                   SYSTEM_PROPERTY_VALUE));
  }

  @Test
  public void whenASystemPropertyCannotBeResolvedAnExceptionIsRaised() {
    expectedException.expect(PropertyNotFoundException.class);
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_UNSET_SYSTEM_PROPERTY), equalTo(SYSTEM_PROPERTY_VALUE));
  }

  @Test
  public void whenNoPropertyIsInTheFileNullValueIsReturned() {
    FileSpanExporterConfiguration fileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    assertThat(fileSpanExporterConfiguration.getStringValue(NO_KEY_IN_FILE), is(nullValue()));
  }

  @Test
  public void whenFileIsNotFoundNoPropertyIsFound() {
    FileSpanExporterConfiguration testNoFileFoundSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_NOT_FOUND_CONF_FILE_NAME);
    assertThat(testNoFileFoundSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), is(nullValue()));
    assertThat(testNoFileFoundSpanExporterConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), is(nullValue()));
  }

  @Test
  public void readsStringValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    String stringValueOne = testFileSpanExporterConfiguration.getStringValue(KEY_STRING_VALUE_ONE);
    String stringValueTwo = testFileSpanExporterConfiguration.getStringValue(KEY_STRING_VALUE_TWO);

    assertThat(stringValueOne, is("stringValueOne"));
    assertThat(stringValueTwo, is("stringValueTwo"));
  }

  @Test
  public void readsBooleanValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    String booleanValue = testFileSpanExporterConfiguration.getStringValue(KEY_BOOLEAN_VALUE);

    assertThat(booleanValue, is("true"));
  }

  @Test
  public void whenValueCorrespondingToPathGetAbsoluteValue() {
    TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, TEST_CONF_FILE_NAME);
    Path caFileLocationPath = testFileSpanExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_EXPORTER_CA_FILE_LOCATION);
    Path keyFileLocationPath = testFileSpanExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_EXPORTER_KEY_FILE_LOCATION);

    assertThat(caFileLocationPath, is(notNullValue()));
    assertThat(keyFileLocationPath, is(notNullValue()));

    assertThat(caFileLocationPath.isAbsolute(), is(TRUE));
    assertThat(keyFileLocationPath.isAbsolute(), is(TRUE));
  }

  @Test
  @Ignore("To be fixed in W-16676258")
  public void configurationFileChanged() throws Exception {
    File file = createTempFile("tracing", "test");
    Path testFile = get(file.getPath());
    URI originalConfigFileUri = getResourceAsUrl(TEST_ORIGINAL_CONF_FILE, getClass()).toURI();
    URI overwrittenConfigFileUri = getResourceAsUrl(TEST_OVERWRITTEN_CONF_FILE, getClass()).toURI();
    copy(get(originalConfigFileUri), testFile, REPLACE_EXISTING);

    final TestFileSpanExporterConfiguration testFileSpanExporterConfiguration =
        new TestFileSpanExporterConfiguration(muleContext, file.getAbsolutePath());

    testFileSpanExporterConfiguration.doOnConfigurationChanged(() -> {
      testFileSpanExporterConfiguration.initialise();
      testFileSpanExporterConfiguration.changed = true;
    });
    assertThat(testFileSpanExporterConfiguration.getStringValue("key"), equalTo("value"));
    copy(get(overwrittenConfigFileUri), testFile, REPLACE_EXISTING);
    new PollingProber(TIMEOUT_MILLIS, DEFAULT_POLLING_INTERVAL)
        .check(new JUnitLambdaProbe(() -> testFileSpanExporterConfiguration.changed));
    assertThat(testFileSpanExporterConfiguration.getStringValue("key"), equalTo("value-overwritten"));
  }

  /**
   * {@link FileSpanExporterConfiguration} used for testing properties file.
   */
  private static class TestFileSpanExporterConfiguration extends FileSpanExporterConfiguration {

    private final String confFileName;
    private boolean changed;

    public TestFileSpanExporterConfiguration(MuleContext muleContext, String confFileName) {
      super(muleContext);
      this.confFileName = confFileName;
    }

    @Override
    protected String getSignalConfigurationFileName() {
      return confFileName;
    }
  }
}
